<?php


namespace Mainto\RpcServer\RpcServer;


use Closure;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Log\LogHandler;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Request\RequestReader;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\Kernel\DebugMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Kernel\ProtectedMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Middleware;
use Mainto\RpcServer\RpcUtil\RpcGCHelper;
use Mainto\RpcServer\Util\IO\Buffer;
use Mainto\RpcServer\Util\Net\Conn;
use Throwable;

class Kernel {
    protected array $middlewares = [

    ];

    /**
     * @var Conn
     */
    protected Conn $conn;
    /**
     * @var RpcStreamContext
     */
    protected RpcStreamContext $requestContext;

    /**
     * @var RpcContext
     */
    protected RpcContext $rpcContext;

    /**
     * @var Buffer
     */
    protected Buffer $bufIO;
    /**
     * @var mixed
     */
    private Closure $pipe;

    public function __construct (Conn $conn, string $streamId) {
        $this->conn = $conn;
        $this->bufIO = new Buffer($conn, 4096, 4096);
        $this->requestContext = new RpcStreamContext($streamId);
        $this->rpcContext = new RpcContext();

        $this->initMiddleware();

        $this->pipe = array_reduce(
            array_reverse($this->middlewares),
            fn(Closure $next, Middleware $middle) => fn(RpcStreamContext $context) => $middle->handle($context, $next),
            fn(RpcStreamContext $context) => $this->invoke($context)
        );
    }

    protected function initMiddleware () {
        if (config('logging.channels')[config('logging.default')]['handler'] == LogHandler::class) {
            if (config('logging.channels')[config('logging.default')]['with']['level'] == "debug") {
                $this->middlewares[] = new DebugMiddleware();
            }
        }

        $this->middlewares = [
            new ProtectedMiddleware(),
        ];
    }

    protected function invoke (RpcStreamContext $context): Response {
        switch ($context->request->getType()) {
            case RequestHeaderType::SystemCheckType:
                return RpcResponseHelper::$cacheResponse->resetBodyAndExt()->useReturnOKType();
            case RequestHeaderType::SidecarStopType:
                Log::debug("exit by recv SidecarStopType signal", ['streamId' => $context->streamId, 'pid' => $context->pid]);
                $context->stop();
                return RpcResponseHelper::$cacheResponse->resetBodyAndExt()->useReturnOKType();
            case RequestHeaderType::InvokeNormalType:
            case RequestHeaderType::UrlInvokeDeleteType:
            case RequestHeaderType::UrlInvokePatchType:
            case RequestHeaderType::UrlInvokePostType:
            case RequestHeaderType::UrlInvokePutType:
            case RequestHeaderType::UrlInvokeGetType:

            case RequestHeaderType::MessageNormalType:
            case RequestHeaderType::MessageDelayType:
            case RequestHeaderType::MessageRetryType:

            case RequestHeaderType::CronNormalType:

            case RequestHeaderType::WebsocketNormalType:
            case RequestHeaderType::WebsocketConnectType:
            case RequestHeaderType::WebsocketDisconnectType:
                return self::invokeToResponse($this->rpcContext->reset($context->request));
        }

        return RpcResponseHelper::$cacheResponse->resetBodyAndExt()->useReturnErrType();
    }

    /**
     * @param RpcContext $context
     * @return Response
     */
    public static function invokeToResponse (RpcContext $context): Response {
        $ret = RpcInvoke::invokeThroughMiddleware($context);

        if ($ret instanceof Response) {
            return $ret;
        }

        // body
        $body = null;
        switch ($context->getMethod()->resSerialize) {
            case 'Json':
            case 'Jsonp':
                $body = RpcResponseHelper::$cacheJsonBody;
                if ($ret instanceof Jsonable) {
                    $encode = (string)$ret->toJson();
                } else {
                    $encode = json_encode($ret instanceof Arrayable ? $ret->toArray() : $ret);

                    if ($encode === false) {
                        throw new RpcRuntimeException("the response can not convert to json: ".json_last_error_msg());
                    }
                }

                $body->setBodyBytes($encode);
                break;
            case 'Raw':
                $body = RpcResponseHelper::$cacheRawBody;
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RpcNotSupportException("body type not support: ".$context->getMethod()->resSerialize);
        }

        $response = RpcResponseHelper::$cacheResponse->resetBodyAndExt()->useReturnOKType();
        $response->setBody($body);

        // responseHeaders
        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        return $response;
    }

    public function run () {
        event('rpc.framework.kernel.starting');

        while (!$this->requestContext->isStopped) {
            if ($this->readAndHandle() === false) {
                break;
            }
            $this->onInvoked();
        }

        $this->conn->close();
        event('rpc.framework.kernel.stopped');
    }

    public function readAndHandle (): bool {
        try {
            $this->requestContext->request = RequestReader::ReadRequestFrom($this->bufIO)->toRequest();
        } catch (Throwable $exception) {
            Log::error($exception, ["streamId" => $this->requestContext->streamId, 'pid' => $this->requestContext->pid]);
            return false;
        }

        $startTime = microtime(true);

        /** @var Response $response */
        $response = ($this->pipe)($this->requestContext);

        $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));
        $response->writeTo($this->bufIO);
        $this->bufIO->flush();
        $response->resetBodyAndExt();

        return true;
    }

    protected function onInvoked () {
        RpcGCHelper::getInstance()->hitGC();
    }
}