<?php


namespace Mainto\RpcServer\RpcServer;


use Closure;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Database\MySqlConnection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\RpcServer\Exceptions\BaseNotReportServiceException;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Request\RequestReader;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\CheckMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\DebugMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Middleware;
use Mainto\RpcServer\RpcServer\Middleware\StopMiddleware;
use Mainto\RpcServer\RpcUtil\RpcGCHelper;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use Mainto\RpcServer\Util\IO\Buffer;
use Mainto\RpcServer\Util\Net\Conn;
use Throwable;

class Kernel {
    protected array $middlewares = [

    ];
    protected ResponseExtendError $cacheExtendError;
    /**
     * @var Response
     */
    protected Response $cacheErrResponse;
    /**
     * @var Conn
     */
    protected Conn $conn;
    /**
     * @var RpcStreamContext
     */
    protected RpcStreamContext $requestContext;
    /**
     * @var Buffer
     */
    protected Buffer $bufIO;
    /**
     * @var mixed
     */
    private Closure $pipe;
    /**
     * @var Response
     */
    private Response $checkOKResponse;

    public function __construct (Conn $conn, string $streamId) {
        $this->cacheExtendError = new ResponseExtendError();
        $this->cacheErrResponse = new Response();
        $this->cacheErrResponse->useReturnErrType();
        $this->cacheExtendError = new ResponseExtendError();
        $this->conn = $conn;
        $this->bufIO = new Buffer($conn, 4096, 4096);
        $this->requestContext = new RpcStreamContext($streamId);

        $this->checkOKResponse = new Response();
        $this->checkOKResponse->useReturnOKType();

        $this->initMiddleware();

        $this->pipe = array_reduce(
            array_reverse($this->middlewares),
            fn($middle, Middleware $next) => fn(RpcStreamContext $context) => $next->handle($context, $middle),
            fn(RpcStreamContext $context) => $this->protectedInvoke($context)
        );
    }

    protected function initMiddleware () {
        $this->middlewares = [
            new StopMiddleware(),
            new CheckMiddleware(),
            new DebugMiddleware(),
        ];
    }

    protected function protectedInvoke (RpcStreamContext $context): Response {
        try {
            return $this->invoke($context);
        } catch (Throwable $exception) {
            /** @var ResponseExtendError $errExt */
            $this->cacheErrResponse->useReturnErrType();
            $this->cacheErrResponse->setExtend($this->cacheExtendError);
            $this->cacheExtendError->setTraceId(isset($request) ? $request->getTraceId() : "");
            $this->cacheExtendError->setErrMsg($exception->getMessage());

            if ($exception instanceof ValidationException) {
                $this->cacheExtendError->setErrorCode(422);
                $this->cacheExtendError->setErrMsg(json_encode($exception->errors()));
            } elseif ($exception instanceof InvalidArgumentException) {
                $this->cacheExtendError->setErrorCode(500);
            } else {
                if (!($exception instanceof BaseNotReportServiceException)) {
                    try {
                        RpcLog::getInstance()->logRequestThrowable($request, $exception, "{$request->getCallClassName()}::{$request->getCallMethodName()}");
                    } catch (Throwable $e) {
                        Log::error($e, ["streamId" => $context->streamId]);
                        // nop
                    }
                }

                $this->cacheExtendError->setErrorCode(intval($exception->getCode()) ?: 500);
            }

            if ($exception instanceof Exception) {
                app(ExceptionHandler::class)->report($exception);
            } else {
                Log::error($exception, ["streamId" => $context->streamId]);
            }

            return $this->cacheErrResponse;
        } finally {
            $this->resourceRecover();
        }
    }

    protected function invoke (RpcStreamContext $context): Response {
        switch ($context->request->getType()) {
            case RequestHeaderType::SystemCheckType:
                return $this->checkOKResponse;
            case RequestHeaderType::InvokeNormalType:
            case RequestHeaderType::UrlInvokeDeleteType:
            case RequestHeaderType::UrlInvokePatchType:
            case RequestHeaderType::UrlInvokePostType:
            case RequestHeaderType::UrlInvokePutType:
            case RequestHeaderType::UrlInvokeGetType:

            case RequestHeaderType::MessageNormalType:
            case RequestHeaderType::MessageDelayType:
            case RequestHeaderType::MessageRetryType:

            case RequestHeaderType::CronNormalType:

            case RequestHeaderType::WebsocketNormalType:
            case RequestHeaderType::WebsocketConnectType:
            case RequestHeaderType::WebsocketDisconnectType:
                return RpcInvoke::invoke($context->request);
        }

        $this->cacheErrResponse->useReturnErrType();
        return $this->cacheErrResponse;
    }

    /**
     * 回收可能导致有问题的资源
     */
    private function resourceRecover () {
        foreach (DB::getConnections() as $connection) {
            if ($connection instanceof MySqlConnection) {
                while ($connection->transactionLevel()) {
                    $connection->rollBack();
                }
            }
        }
    }

    public function run () {
        event('rpc.framework.kernel.starting');

        while (!$this->requestContext->isStopped) {
            if ($this->readAndHandle() === false) {
                break;
            }
            $this->onInvoked();
        }

        $this->conn->close();
        event('rpc.framework.kernel.stopped');
    }

    public function readAndHandle (): bool {
        try {
            $this->requestContext->request = RequestReader::ReadRequestFrom($this->bufIO)->toRequest();
        } catch (Throwable $exception) {
            Log::error($exception, ["streamId" => $this->requestContext->streamId, 'pid' => $this->requestContext->pid]);
            return false;
        }

        $startTime = microtime(true);

        $response = ($this->pipe)($this->requestContext);

        $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));
        $response->writeTo($this->bufIO);
        $this->bufIO->flush();
        $response->resetBodyAndExt();

        return true;
    }

    protected function onInvoked () {
        RpcGCHelper::getInstance()->hitGC();
    }
}