<?php


namespace Mainto\RpcServer\RpcServer\Middleware;


use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\Exceptions\BaseNotReportServiceException;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\RpcStreamContext;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use Psr\SimpleCache\InvalidArgumentException;
use Throwable;

class ProtectedMiddleware implements Middleware {
    protected ResponseExtendError $cacheExtendError;
    /**
     * @var Response
     */
    protected Response $cacheErrResponse;

    public function __construct () {
        $this->cacheExtendError = new ResponseExtendError();
        $this->cacheErrResponse = new Response();
        $this->cacheErrResponse->useReturnErrType();
        $this->cacheExtendError = new ResponseExtendError();
    }

    public function handle (RpcStreamContext $context, \Closure $next): Response {
        try {
            return $next($context);
        } catch (Throwable $exception) {
            /** @var ResponseExtendError $errExt */
            $this->cacheErrResponse->useReturnErrType();
            $this->cacheErrResponse->setExtend($this->cacheExtendError);
            $this->cacheExtendError->setTraceId(isset($request) ? $request->getTraceId() : "");
            $this->cacheExtendError->setErrMsg($exception->getMessage());

            if ($exception instanceof ValidationException) {
                $this->cacheExtendError->setErrorCode(422);
                $this->cacheExtendError->setErrMsg(json_encode($exception->errors()));
            } elseif ($exception instanceof InvalidArgumentException) {
                $this->cacheExtendError->setErrorCode(500);
            } else {
                if (!($exception instanceof BaseNotReportServiceException)) {
                    try {
                        RpcLog::getInstance()->logRequestThrowable($request, $exception, "{$request->getCallClassName()}::{$request->getCallMethodName()}");
                    } catch (Throwable $e) {
                        Log::error($e, ["streamId" => $context->streamId]);
                        // nop
                    }
                }

                $this->cacheExtendError->setErrorCode(intval($exception->getCode()) ?: 500);
            }

            if ($exception instanceof Exception) {
                app(ExceptionHandler::class)->report($exception);
            } else {
                Log::error($exception, ["streamId" => $context->streamId]);
            }

            return $this->cacheErrResponse;
        }
    }
}