<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Log\LogHandler;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\DebugMiddleware;
use Mainto\RpcServer\Util\Net\Conn;
use Mainto\RpcServer\Util\Net\Net;
use Mainto\RpcServer\Util\OS;
use RuntimeException;
use Throwable;

class MasterKernel extends Kernel {
    public function __construct (Conn $conn, string $streamId) {
        parent::__construct($conn, $streamId);

        app('events')->listen('rpc.framework.kernel.starting', function () {
            pcntl_signal(SIGCHLD, SIG_IGN);
            event('rpc.framework.kernel.master.starting');
        });
    }

    protected function initMiddleware () {
        $this->middlewares = [
            new DebugMiddleware(),
        ];
    }

    protected function invoke (RpcStreamContext $context): Response {
        switch ($context->request->getType()) {
            case RequestHeaderType::SystemCheckType:
                return $this->cacheOKResponse;
            case RequestHeaderType::SidecarConfigType:
                return self::serviceConfigResponse();
            case RequestHeaderType::SidecarStopType:
                Log::debug("exit by recv SidecarStopType signal", ['streamId' => $context->streamId, 'pid' => $context->pid]);
                $context->stop();
                return $this->cacheOKResponse;
            case RequestHeaderType::SidecarScaleUpType:
                $unixAddress = collect($context->request->getBody()->getContent())->get("unix");
                $childId = pcntl_fork();
                if ($childId < 0) {
                    throw new RuntimeException("fork fail");
                } elseif ($childId == 0) {
                    try {
                        LogHandler::init(storage_path("logs/service/".now()->format("Y-m-d")), pathinfo($unixAddress, PATHINFO_FILENAME));

                        Log::info("worker start and serve", ['pid' => getmypid(), 'unix' => $unixAddress]);

                        //                    $this->close();
                        OS::setProcessTitle(sprintf("rpc-php [%s] [%s] child", config('rpc-server.service_name'), $unixAddress));

                        $seed = intval(time() + (microtime(true) * 1000));
                        mt_srand($seed);

                        app(Kernel::class, [
                            'conn'     => Net::dialTimeout("unix", $unixAddress, 3),
                            'streamId' => $unixAddress,
                        ])->run();
                    } catch (Throwable $exception) {
                        Log::warning("child exit by error", ['error' => $exception->getMessage()]);
                    } finally {
                        exit(0);
                    }
                }

                $this->cacheOKResponse->useReturnOKType();
                $this->cacheOKResponse->setBody(Body::newJsonBody(json_encode(["pid" => $childId])));
                return $this->cacheOKResponse;
            default:
                return $this->cacheErrResponse;
        }
    }

    public static function serviceConfigResponse (): Response {
        $response = new Response();
        $response->useReturnOKType();

        $body = Body::newJsonBody(json_encode(RpcRouter::getInstance()->getBaseRouterConfig()));
        $response->setBody($body);

        return $response;
    }
}