<?php


namespace Mainto\RpcServer\RpcServer;


use Closure;
use Illuminate\Database\MySqlConnection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Request\RequestReader;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\DebugMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Middleware;
use Mainto\RpcServer\RpcServer\Middleware\ProtectedMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\RecoverMiddleware;
use Mainto\RpcServer\RpcUtil\RpcGCHelper;
use Mainto\RpcServer\Util\IO\Buffer;
use Mainto\RpcServer\Util\Net\Conn;
use Throwable;

class Kernel {
    protected array $middlewares = [

    ];
    protected ResponseExtendError $cacheExtendError;
    /**
     * @var Response
     */
    protected Response $cacheErrResponse;
    /**
     * @var Conn
     */
    protected Conn $conn;
    /**
     * @var RpcStreamContext
     */
    protected RpcStreamContext $requestContext;
    /**
     * @var Buffer
     */
    protected Buffer $bufIO;
    /**
     * @var Response
     */
    protected Response $cacheOKResponse;
    /**
     * @var mixed
     */
    private Closure $pipe;

    public function __construct (Conn $conn, string $streamId) {
        $this->cacheExtendError = new ResponseExtendError();
        $this->cacheErrResponse = new Response();
        $this->cacheErrResponse->useReturnErrType();
        $this->cacheExtendError = new ResponseExtendError();
        $this->conn = $conn;
        $this->bufIO = new Buffer($conn, 4096, 4096);
        $this->requestContext = new RpcStreamContext($streamId);

        $this->cacheOKResponse = new Response();
        $this->cacheOKResponse->useReturnOKType();

        $this->initMiddleware();

        $this->pipe = array_reduce(
            array_reverse($this->middlewares),
            fn(Closure $next, Middleware $middle) => fn(RpcStreamContext $context) => $middle->handle($context, $next),
            fn(RpcStreamContext $context) => $this->invoke($context)
        );
    }

    protected function initMiddleware () {
        $this->middlewares = [
            new DebugMiddleware(),
            new ProtectedMiddleware(),
            new RecoverMiddleware(),
        ];
    }

    protected function invoke (RpcStreamContext $context): Response {
        switch ($context->request->getType()) {
            case RequestHeaderType::SystemCheckType:
                return $this->cacheOKResponse;
            case RequestHeaderType::SidecarStopType:
                Log::debug("exit by recv SidecarStopType signal", ['streamId' => $context->streamId, 'pid' => $context->pid]);
                $context->stop();
                return $this->cacheOKResponse;
            case RequestHeaderType::InvokeNormalType:
            case RequestHeaderType::UrlInvokeDeleteType:
            case RequestHeaderType::UrlInvokePatchType:
            case RequestHeaderType::UrlInvokePostType:
            case RequestHeaderType::UrlInvokePutType:
            case RequestHeaderType::UrlInvokeGetType:

            case RequestHeaderType::MessageNormalType:
            case RequestHeaderType::MessageDelayType:
            case RequestHeaderType::MessageRetryType:

            case RequestHeaderType::CronNormalType:

            case RequestHeaderType::WebsocketNormalType:
            case RequestHeaderType::WebsocketConnectType:
            case RequestHeaderType::WebsocketDisconnectType:
                return RpcInvoke::invoke($context->request);
        }

        $this->cacheErrResponse->useReturnErrType();
        return $this->cacheErrResponse;
    }

    public function run () {
        event('rpc.framework.kernel.starting');

        while (!$this->requestContext->isStopped) {
            if ($this->readAndHandle() === false) {
                break;
            }
            $this->onInvoked();
        }

        $this->conn->close();
        event('rpc.framework.kernel.stopped');
    }

    public function readAndHandle (): bool {
        try {
            $this->requestContext->request = RequestReader::ReadRequestFrom($this->bufIO)->toRequest();
        } catch (Throwable $exception) {
            Log::error($exception, ["streamId" => $this->requestContext->streamId, 'pid' => $this->requestContext->pid]);
            return false;
        }

        $startTime = microtime(true);

        $response = ($this->pipe)($this->requestContext);

        $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));
        $response->writeTo($this->bufIO);
        $this->bufIO->flush();
        $response->resetBodyAndExt();

        return true;
    }

    protected function onInvoked () {
        RpcGCHelper::getInstance()->hitGC();
    }

    /**
     * 回收可能导致有问题的资源
     */
    private function resourceRecover () {
        foreach (DB::getConnections() as $connection) {
            if ($connection instanceof MySqlConnection) {
                while ($connection->transactionLevel()) {
                    $connection->rollBack();
                }
            }
        }
    }
}