<?php


namespace Mainto\RpcServer\Util\ObjectMapper;


use Illuminate\Support\Str;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\Types\Map;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use ReflectionProperty;

class ClassRef implements PropertiesInterface {
    public function getProperties (): array {
        return $this->properties;
    }
    /**
     * @var PropertyRef[]
     */
    public array $properties;

    /**
     * @var string[]
     */
    public array $useClasses;
    public string $namespace;

    public ReflectionClass $class;
    public string $name;

    /**
     * ClassRef constructor.
     * @param $object
     * @throws ReflectionException
     */
    public function __construct ($object) {
        $this->class = new ReflectionClass($object);

        $this->useClasses = self::parseUseClasses($this->class->getFileName());
        $this->namespace = $this->class->getNamespaceName();
        $this->name = $this->class->getShortName();
        $this->loadProperties();
    }

    public function loadProperties() {
        $methods = array_flip(array_map(fn($method) => $method->name, $this->class->getMethods(ReflectionMethod::IS_PUBLIC)));

        $this->properties = array_map(function (ReflectionProperty $reflectionProperty) use ($methods) {
            $setMethod = 'set'.Str::studly($reflectionProperty->name);

            $property = new PropertyRef();
            [$property->type, $_] = $this->getType($reflectionProperty, $this);
            $property->name = $reflectionProperty->name;
            $property->setterMethod = isset($methods[$setMethod]) ? $setMethod : '';

            return $property;
        }, $this->class->getProperties(ReflectionProperty::IS_PUBLIC));
    }

    public static function parseUseClasses ($classFile): array {
        $useClasses = [];

        $fp = fopen($classFile, 'r');
        while (($line = ltrim(fgets($fp))) !== false) {
            if (starts_with($line, 'class')) {
                break;
            }
            if (starts_with($line, 'use')) {
                $useStr = rtrim(ltrim($line, 'use '), "; \n");
                if (str_contains($useStr, 'as')) {
                    [$fullClass, $useName] = explode('as', $useStr);
                    $useClasses[trim($useName)] = trim($fullClass);
                } else {
                    $useClasses[class_basename($useStr)] = $useStr;
                }
            }
        }
        fclose($fp);

        return $useClasses;
    }

    protected function getType (ReflectionProperty $property, self $classRef) {
        if ($property->getType()) {
            $type = $property->getType()->getName();
            switch ($type) {
                case 'array':
                    return $this->parseArrayType($property, $classRef);
                case Map::class:
                    return $this->parseMapType($property, $classRef);
                default:
                    return [$type, $type];
            }
        }

        return ['mixed', null];
    }

    protected function parseMapType (ReflectionProperty $property, self $classRef): array {
        $doc = $property->getDocComment();
        if (!$doc) return ['Map::', null];

        foreach (explode("\n", $doc) as $line) {
            if (str_contains($line, '@var')) {
                $match = [];
                $type = null;
                $keyType = 'string|int';
                if (preg_match('/.*@var\s*Map<(.*)>/', $line, $match) && count($match) > 1) {
                    $types = explode(',', $match[1]);
                    $type = trim(array_pop($types));
                    $keyType = trim(array_pop($types) ?: $keyType);
                }

                /*** @var Map $_ */
                if (preg_match('/.*@var\s*\\\Mainto\\\RpcServer\\\Util\\\Types\\\Map<(.*)>/', $line, $match) && count($match) > 1) {
                    $types = explode(',', $match[1]);
                    $type = trim(array_pop($types));
                    $keyType = trim(array_pop($types) ?: $keyType);
                }

                if ($type !== null) {
                    if (in_array($type, Language::$simpleType)) {
                        return ["Map<$keyType, $type>", $type];
                    }
                    if (isset($classRef->useClasses[$type])) {
                        return ["Map<$keyType, {$classRef->useClasses[$type]}>", $classRef->useClasses[$type]];
                    }

                    return [sprintf("Map<$keyType, %s>", $classRef->namespace.'\\'.$type), $classRef->namespace.'\\'.$type];
                }

                break;
            }
        }

        return ['Map::', null];
    }

    protected function parseArrayType (ReflectionProperty $property, self $classRef): array {
        $doc = $property->getDocComment();
        if (!$doc) return ['array', null];

        foreach (explode("\n", $doc) as $line) {
            if (str_contains($line, '@var')) {
                $match = [];
                $type = null;
                if (preg_match('/.*@var\s*(.*)\[]/', $line, $match) && count($match) > 1) {
                    $type = $match[1];
                }
                $arrayDeepCount = 1;
                while ($type && Str::endsWith($type, '[]')) {
                    $type = substr($type, 0, -2);
                    $arrayDeepCount ++;
                }

                if (preg_match('/.*@var\s*array<(.*)>/', $line, $match) && count($match) > 1) {
                    $types = explode(',', $match[1]);
                    $type = array_pop($types);
                }

                if ($type !== null) {
                    if (in_array($type, Language::$simpleType)) {
                        return [$type.str_repeat('[]', $arrayDeepCount), $type];
                    }
                    if (isset($classRef->useClasses[$type])) {
                        return [$classRef->useClasses[$type].str_repeat('[]', $arrayDeepCount), $classRef->useClasses[$type]];
                    }

                    return [$classRef->namespace.'\\'.$type.str_repeat('[]', $arrayDeepCount), $classRef->namespace.'\\'.$type];
                }

                break;
            }
        }

        return ['array', null];
    }
}