<?php

namespace Mainto\RpcServer\RpcUtil\Tool\MonitorDriver;

use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\RpcClient\RpcClient;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcUtil\Tool\RpcMonitorInterface;
use RuntimeException;
use Throwable;

class SidecarPrometheus implements RpcMonitorInterface {
    private static ?RpcClient $rpcClient = null;

    private static ?Request $cacheRequest = null;

    private static array $actionMap = [
        'set' => ['type'=> 'gauge', 'action'=> 'Set'],
        'add' => ['type'=> 'gauge', 'action'=> 'Add'],
        'sub' => ['type'=> 'gauge', 'action'=> 'Sub'],
        'inc' => ['type'=> 'gauge', 'action'=> 'Inc'],
        'dec' => ['type'=> 'gauge', 'action'=> 'Dec'],
        'summary' => ['type'=> 'summary', 'action'=> 'Observe'],
    ];

    public function __construct () {
        if (self::$cacheRequest === null) {
            $requestExtendHeader = new RequestExtendHeader();
            self::$cacheRequest = new Request();
            self::$cacheRequest->setType(RequestHeaderType::SidecarMonitorType);
            self::$cacheRequest->setExtend($requestExtendHeader);
        }
    }

    /**
     * send message to sidecar sls service
     *
     * @param $source
     * @param $logStore
     * @param $contents
     */
    protected function send (array $contents) {
        try {
            /** @var RequestExtendHeader $ext */
            $ext = self::$cacheRequest->getExt();
            $contents['service_name'] = config('rpc-server.service_name');
            self::$cacheRequest->setBody(Body::newJsonBody(json_encode($contents)));

            if (self::$rpcClient === null) {
                self::$rpcClient = RpcClient::getMonitorClient();
            }
            self::$cacheRequest->setTraceId(RpcContext::$currentTraceId);
            $response = self::$rpcClient->Do(self::$cacheRequest);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $e) {
            (optional(self::$rpcClient))->close();
            self::$rpcClient = null;
            Log::error($e, $contents);
            throw new RuntimeException("can not find this monitor server", 0, $e);
        }
    }

    private function dealParam (string $function, string $name, float $value = null) {
        if (! self::$actionMap[$function]) throw new RuntimeException("actionMap not exist", 0);
        return [
            "type"   => data_get(self::$actionMap[$function], 'type'),
            "name"   => $name,
            "content" => [
                'action' => data_get(self::$actionMap[$function], 'action'),
                'value' => $value
            ],
        ];
    }

    public function set (string $name, float $value) {
        $this->send($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function add (string $name, float $value) {
        $this->send($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function sub (string $name, float $value) {
        $this->send($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function inc (string $name) {
        $this->send($this->dealParam(__FUNCTION__, $name));
    }

    public function dec (string $name) {
        $this->send($this->dealParam(__FUNCTION__, $name));
    }

    public function summary (string $name, float $value) {
        $this->send($this->dealParam(__FUNCTION__, $name, $value));
    }
}
