<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Contracts\Support\Arrayable;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\Enum;
use Mainto\RpcServer\RpcServer\Definition\RpcObject\JsonMapper;
use Mainto\RpcServer\RpcServer\Definition\RpcObject\ObjectRef;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\Util\ArrayHelper;
use Mainto\RpcServer\Util\ObjectMapper\MapperInterface;
use RuntimeException;

class RpcDefinition implements Arrayable {
    use ArrayHelper;

    public const EmptyType = '#empty';
    public const SimpleTypes = [
        "string",
        "bool",
        "int",
        "uint",
        "double",
        "unsignedDouble",
        "array",
        "array<int>",
        "array<uint>",
        "array<string>",
    ];
    private static ?RpcDefinition $instance = null;
    /**
     * @var Controller[]
     */
    public array $controllers = [];
    /**
     * @var Struct[]
     */
    public array $structs = [];

    /**
     * @var Enum[]
     */
    public array $enums = [];
    /**
     * @var Struct[] [typeName => Struct]
     */
    private array $structMap = [];
    private MapperInterface $mapper;

    private function __construct () {
        $empty = new Struct();
        $empty->name = "EmptyStruct";
        $empty->namespace = "EmptyStruct";

        $this->structs[] = $this->structMap[self::EmptyType] = $empty;
        $this->mapper = new JsonMapper();
    }

    public static function getInstance (): RpcDefinition {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    public static function newInstance (): RpcDefinition {
        return new self();
    }

    public static function clear () {
        self::$instance = null;
    }

    public function inStructCache (string $type) {
        return isset($this->structMap[$type]);
    }

    public function getStruct (string $type): Struct {
        if (!isset($this->structMap[$type])) {
            throw new RuntimeException("the struct not exists");
        }

        return $this->structMap[$type];
    }

    public function objectStruct (string $type): Struct {
        if (!isset($this->structMap[$type])) {
            $this->structMap[$type] = $this->structs[] = $struct = new Struct();

            $objectRef = ObjectRef::getRef($type);
            $struct->name = $objectRef->name;
            $struct->namespace = $objectRef->namespace;
            $struct->properties = $objectRef->properties;
        }

        return $this->structMap[$type];
    }

    public function notDefinitionStruct (string $type): Struct {
        if (!isset($this->structMap[$type])) {
            $this->structMap[$type] = $this->structs[] = $struct = new Struct();
            $struct->name = $type;
            $struct->namespace = $type;
        }

        return $this->structMap[$type];
    }

    public function parse (string $path) {
        $this->mapper->map(json_decode(file_get_contents($path), true), $this);
    }

    public function getMapper () {
        return $this->mapper;
    }
}