<?php


namespace Mainto\RpcServer\Util\ObjectMapper;


use Illuminate\Support\Str;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use ReflectionProperty;

class ClassRef {
    /**
     * @var PropertyRef[]
     */
    public array $publicPropertyRefs;
    public array $useClasses;
    public string $namespace;

    public ReflectionClass $class;

    /**
     * ClassRef constructor.
     * @param $object
     * @throws ReflectionException
     */
    public function __construct ($object) {
        $this->class = new ReflectionClass($object);

        $this->useClasses = $this->parseUseClasses($this->class->getFileName());
        $this->namespace = $this->class->getNamespaceName();

        $methods = array_flip(array_map(fn($method) => $method->name, $this->class->getMethods(ReflectionMethod::IS_PUBLIC)));

        $this->publicPropertyRefs = array_map(function (ReflectionProperty $property) use ($methods) {
            $setMethod = 'set'.Str::studly($property->name);

            return new PropertyRef($property, $this->getType($property, $this), isset($methods[$setMethod]) ? $setMethod : '');
        }, $this->class->getProperties(ReflectionProperty::IS_PUBLIC));
    }

    private function parseUseClasses ($classFile) {
        $useClasses = [];

        $fp = fopen($classFile, 'r');
        while (($line = ltrim(fgets($fp))) !== false) {
            if (str_starts_with($line, 'class')) {
                break;
            }
            if (str_starts_with($line, 'use')) {
                $useStr = rtrim(ltrim($line, 'use '), "; \n");
                if (str_contains($useStr, 'as')) {
                    [$fullClass, $useName] = explode('as', $useStr);
                    $useClasses[trim($useName)] = trim($fullClass);
                } else {
                    $useClasses[class_basename($useStr)] = $useStr;
                }
            }
        }
        fclose($fp);

        return $useClasses;
    }

    protected function getType (ReflectionProperty $property, ClassRef $classRef) {
        $type = $property->getType()->getName();
        switch ($type) {
            case 'array':
                return $this->parseArrayType($property, $classRef);
            default:
                return $type;
        }
    }

    protected function parseArrayType (ReflectionProperty $property, ClassRef $classRef) {
        $doc = $property->getDocComment();
        if (!$doc) return 'array';

        foreach (explode("\n", $doc) as $line) {
            if (str_contains($line, '@var')) {
                $match = [];
                preg_match('/.*@var\s*(.*)\[]/', $line, $match);
                if (count($match) > 1) {
                    if (isset($classRef->useClasses[$match[1]])) {
                        return $classRef->useClasses[$match[1]].'[]';
                    }

                    return $classRef->namespace.'\\'.$match[1].'[]';
                }
                break;
            }
        }

        return 'array';
    }
}