<?php


namespace Mainto\RpcServer\RpcServer\Definition\RpcObject;


use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcAnnotations\RpcComment;
use Mainto\RpcServer\RpcAnnotations\RpcNullable;
use Mainto\RpcServer\RpcAnnotations\RpcRequired;
use Mainto\RpcServer\RpcAnnotations\RpcValidation;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Traits\ValidateGetter;
use Mainto\RpcServer\RpcServer\RpcObject;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use ReflectionProperty;

class ObjectRef {
    use ValidateGetter;

    /**
     * @var array
     */
    private static array $classRefMap;
    /**
     * @var Property[]
     */
    public array $properties;
    public array $useClasses;
    public string $namespace;
    public string $name;

    public ReflectionClass $class;

    /**
     * @param $object
     * @return ObjectRef
     * @throws ReflectionException
     */
    public static function getRef($object): ObjectRef {
        $objectKey = is_object($object) ? get_class($object): $object;
        if (isset(self::$classRefMap[$objectKey])) {
            return self::$classRefMap[$objectKey];
        } else {
            return self::$classRefMap[$objectKey] = new self($object);
        }
    }

    /**
     * ClassRef constructor.
     * @param $object
     * @throws ReflectionException
     */
    public function __construct ($object) {
        $this->class = new ReflectionClass($object);
        $this->name = $this->class->name;
        $this->namespace = $this->class->getNamespaceName();
        $this->useClasses = $this->parseUseClasses($this->class->getFileName());

        $methods = array_flip(array_map(fn($method) => $method->name, $this->class->getMethods(ReflectionMethod::IS_PUBLIC)));

        // todo in php8 could use https://www.php.net/manual/en/reflectionproperty.getdefaultvalue.php
        $defaultProperties = $this->class->getDefaultProperties();

        $this->properties = array_map(function (ReflectionProperty $property) use ($methods, $defaultProperties) {
            $setMethod = 'set'.Str::studly($property->name);

            $definitionProperty = new Property();
            $definitionProperty->name = $property->name;
            $definitionProperty->type = $this->getType($property, $this);
            if (is_subclass_of($definitionProperty->type, RpcObject::class)) {
                $_ = RpcDefinition::getInstance()->objectStruct($definitionProperty->type);
            }

            foreach (app(AnnotationReader::class)->getPropertyAnnotations($property) as $annotation) {
                if ($annotation instanceof RpcRequired) {
                    $definitionProperty->require = true;
                } elseif ($annotation instanceof RpcComment) {
                    $definitionProperty->comment = $annotation->comment;
                } elseif ($annotation instanceof RpcValidation) {
                    $definitionProperty->validation = $annotation->validation;
                } elseif ($annotation instanceof RpcNullable) {
                    $definitionProperty->nullable = true;
                }
            }

            if (isset($defaultProperties[$property->name])) {
                $definitionProperty->default = $defaultProperties[$property->name];
            }

            $definitionProperty->setterMethod = isset($methods[$setMethod]) ? $setMethod : '';

            return $definitionProperty;
        }, $this->class->getProperties(ReflectionProperty::IS_PUBLIC));
    }

    private function parseUseClasses ($classFile) {
        $useClasses = [];

        $fp = fopen($classFile, 'r');
        while (($line = ltrim(fgets($fp))) !== false) {
            if (str_starts_with($line, 'class')) {
                break;
            }
            if (str_starts_with($line, 'use')) {
                $useStr = rtrim(ltrim($line, 'use '), "; \n");
                if (str_contains($useStr, 'as')) {
                    [$fullClass, $useName] = explode('as', $useStr);
                    $useClasses[trim($useName)] = trim($fullClass);
                } else {
                    $useClasses[class_basename($useStr)] = $useStr;
                }
            }
        }
        fclose($fp);

        return $useClasses;
    }

    protected function getType (ReflectionProperty $property, self $classRef) {
        $type = $property->getType()->getName();
        switch ($type) {
            case 'array':
                return $this->parseArrayType($property, $classRef);
            default:
                return $type;
        }
    }

    protected function parseArrayType (ReflectionProperty $property, self $classRef) {
        $doc = $property->getDocComment();
        if (!$doc) return 'array';

        foreach (explode("\n", $doc) as $line) {
            if (str_contains($line, '@var')) {
                $match = [];
                preg_match('/.*@var\s*(.*)\[]/', $line, $match);
                if (count($match) > 1) {
                    if (isset($classRef->useClasses[$match[1]])) {
                        return $classRef->useClasses[$match[1]].'[]';
                    }

                    return $classRef->namespace.'\\'.$match[1].'[]';
                }
                break;
            }
        }

        return 'array';
    }
}