<?php

namespace Mainto\RpcServer\RpcServer;

use Doctrine\Common\Annotations\AnnotationReader;
use InvalidArgumentException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcAnnotations\RpcWebsocket;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\Definition;
use Mainto\RpcServer\RpcServer\Definition\Enum;
use Mainto\RpcServer\RpcServer\Definition\Method;
use Mainto\RpcServer\Service\Struct\Config;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class RpcRouter {

    /**
     * @var RpcRouter
     */
    private static ?RpcRouter $instance = null;

    /**
     * 注解解释器
     * @var AnnotationReader
     */
    private AnnotationReader $_annotationReader;

    /**
     * alias Map
     * @var array ['controller' => 'alias']
     */
    private array $aliasMap = [];

    /**
     * @var array ['alias' => controller]
     */
    private array $controllerMap = [];

    /**
     * @var array ['alias_methodName' => method]
     */
    private array $controllerMethodMap = [];

    public static string $enumsPath = "";

    /**
     * RpcRouter constructor.
     */
    private function __construct () {
        $this->_annotationReader = new AnnotationReader();
    }

    public static function getInstance () {
        if (self::$instance == null) {
            self::$instance = new RpcRouter();
            self::$instance->registerEnums();
        }

        return self::$instance;
    }

    public static function clear() {
        self::$instance = null;
    }

    public function registerEnums() {
        if (!self::$enumsPath) {
            self::$enumsPath = app_path('Enums');
        }

        if (!file_exists(self::$enumsPath)) return;

        $enumFiles = get_files(self::$enumsPath);
        foreach ($enumFiles as $_name) {
            if (substr($_name, -8, 8) != 'Enum.php') {
                continue;
            }

            $namespace = null;

            if (preg_match('#^namespace\s+(.+?);$#sm', file_get_contents($_name), $m)) {
                $namespace = $m[1];
            }
            if (!$namespace) {
                throw new RuntimeException(sprintf("get namespace fail from %s", $_name));
            }

            $_nameArr = explode('/', $_name);
            $className = str_replace(['/', '.php'], ['\\', ''], array_pop($_nameArr));

            $enum = $namespace.'\\'.$className;

            $reflection = new ReflectionClass($enum);

            $enum = new Enum();
            $enum->namespace = $reflection->getNamespaceName();
            $enum->name = $reflection->getName();
            $enum->enums = $reflection->getConstants();

            Definition::getInstance()->enums[] = $enum;
        }
    }

    /**
     * 注册一个RPC路由
     *
     * @param string $class
     * @param string|null $alias
     */
    public function register (string $class, string $alias = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$alias) {
                $alias = $refClass->getName();
                $pos = strrpos($alias, 'Controllers\\');
                if ($pos !== false) {
                    $alias = substr($alias, $pos + 12);
                } else {
                    throw new RuntimeException($class.' not has alias and path error');
                }
            }

            $this->addController($refClass, $alias);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    /**
     * @param ReflectionClass $refClass
     * @param string $alias
     * @throws ReflectionException
     */
    public function addController (ReflectionClass $refClass, string $alias) {
        if (isset($this->controllerMap[$alias])) {
            if ($refClass->isInstance($this->controllerMap[$alias]->instance)) {
                return;
            } else {
                throw new InvalidArgumentException("duplicate class name/alias: {$alias}");
            }
        }

        if (!ends_with($alias, 'Controller')) {
            throw new RuntimeException("class name must be use Controller end");
        }

        $definition = Definition::getInstance();
        $definition->controllers[] = $controller = new Controller($refClass->newInstance());
        $controller->alias = $alias;
        $controller->shortName = substr($alias, 0, -10);;
        $controller->name = $refClass->getName();

        if (!($controller->getInstance() instanceof \Mainto\RpcServer\Base\Controller)) {
            throw new RuntimeException("register router {$refClass->getName()} is not support");
        }

        $methods = $refClass->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod($controller, $method);
        }

        // cache for fast find
        $this->controllerMap[$alias] = $controller;
        $this->aliasMap[$alias] = $refClass->getName();
    }

    /**
     * @param Controller $controller
     * @param ReflectionMethod $refMethod
     * @throws ReflectionException
     */
    public function addMethod (Controller $controller, ReflectionMethod $refMethod) {
        $methodName = $refMethod->getShortName();
        if ($methodName[0] == "_") {
            return;
        }

        $controller->methods[] = $method = new Method($refMethod);
        $method->name = $methodName;

        $defaultResponse = new Response();
        $defaultResponse->useReturnOKType();
        $defaultResponse->setBody(Body::newJsonBody("{}"));

        $annotations = app(AnnotationReader::class)->getMethodAnnotations($refMethod);
        foreach ($annotations as $annotation) {
            if ($annotation instanceof RpcApi) {
                $method->loadFromRpcApi($annotation);
            } elseif ($annotation instanceof RpcHeader) {
                $method->loadFromRpcHeader($annotation);
            } elseif ($annotation instanceof RpcFormat) {
                $method->loadFromRpcFormat($annotation);
            } elseif ($annotation instanceof RpcResponseHeader) {
                $method->loadFromRpcResponseHeader($annotation);
            } elseif ($annotation instanceof RpcAuthority) {
                $method->loadFromRpcAuthority($annotation);
            } elseif ($annotation instanceof RpcMessageHook) {
                $method->loadFromRpcMessageHook($annotation);
            } elseif ($annotation instanceof RpcWebsocket) {
                $method->loadFromRpcWebsocket($annotation);
            } elseif ($annotation instanceof RpcDisableTypeValidation) {
                $method->loadFromRpcDisableTypeValidation($annotation);
            } elseif ($annotation instanceof RpcCron) {
                $method->loadFromRpcCron($annotation, $controller->alias);
            } elseif ($annotation instanceof RpcParam) {
                $method->loadFromRpcParam($annotation);
            }
        }

        // validate and cache
        $_ = $method->getParameters();

        $this->controllerMethodMap[$controller->alias][$method->name] = $method;
    }

    public function getController (string $fullClassName): ?Controller {
        return $this->controllerMap[$fullClassName] ?? null;
    }

    public function getControllerMethod (string $fullClassName, $methodName): ?Method {
        return $this->controllerMethodMap[$fullClassName][$methodName] ?? null;
    }

    public function getConfig (): Config {
        return Config::createFromDefinition(Definition::getInstance());
    }

    /**
     * 获取控制器名称
     *
     * @param $class
     * @return string|null
     */
    public function getControllerAlias ($class) {
        return $this->aliasMap[$class] ?? null;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public function getBaseRouterConfig () {
        $struct = Config::createFromDefinition(Definition::getInstance());

        $config = [
            'name'       => config('rpc-server.service_name'),
            'paths'      => $struct->getPaths()->toArray(),
            'crons'      => [],
            'messages'   => [],
            'websockets' => $struct->getWebsockets()->toArray(),
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = $struct->getCrons()->toArray();
        }

        if (config('rpc-server.sidecar.rabbitmq.enable')) {
            $config['messages'] = $struct->getMessages()->toArray();
        }

        return $config;
    }
}