<?php


namespace Mainto\RpcServer\RpcServer\Definition;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcAnnotations\RpcWebsocket;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcObject;
use Mainto\RpcServer\Util\ArrayHelper;
use Mainto\RpcServer\Util\Language;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class Method implements Arrayable {
    use ArrayHelper;

    /**
     * @var string
     */
    public string $name;

    /**
     * @var string
     */
    public string $alias = "";

    /**
     * @var bool
     */
    public bool $deprecated = false;

    /**
     * @var string
     */
    public string $description = "";

    /**
     * @var HttpApi|null
     */
    public ?HttpApi $httpApi = null;

    /**
     * @var array<Header>
     */
    public array $headers = [];

    /**
     * @var string
     */
    public string $requestType = RpcDefinition::EmptyType;

    /**
     * @var object|null
     */
    public ?object $responseType = null;

    /**
     * @var string
     */
    public string $resSerialize = "Json";

    /**
     * @var array<ResponseHeader>
     */
    public array $resHeaders = [];

    /**
     * @var bool
     */
    public bool $checkValidation = true;

    /**
     * @var bool
     */
    public bool $checkType = true;

    /**
     * @var array|null
     */
    public array $needAuthorities = [];

    /**
     * @var string|null
     */
    public ?string $wsRoom = null;

    /**
     * @var string|null
     */
    public ?string $wsNeedConnectMessage = null;

    /**
     * @var string|null
     */
    public ?string $wsNeedDisconnectMessage = null;

    /**
     * @var string|null
     */
    public ?string $messageHookName = null;

    /**
     * @var string|null
     */
    public ?string $cronSpec = null;

    /**
     * @var string|null
     */
    public ?string $cronHookName = null;

    private ReflectionMethod $refInstance;

    /**
     * @var array|null [name]
     */
    private ?array $requestDeclareMap = null;

    /**
     * @var array|null [name]
     */
    private ?array $requestDeclareNames = null;

    /**
     * @var array|null [name => Property]
     */
    private ?array $parameters = null;

    public function __construct (ReflectionMethod $refInstance) {
        $this->refInstance = $refInstance;
    }

    /**
     * @param array $params
     * @throws ValidationException
     */
    public function validate (array &$params) {
        $requestType = RpcDefinition::getInstance()->getStruct($this->requestType);

        $this->handlerDefaultAndType($requestType, $params);

        $requestType->validate($params);
    }

    private function handlerDefaultAndType (Struct $requestType, array &$params) {
        foreach ($requestType->properties as $property) {
            [$exists, $value] = $this->keyExists($params, $property->name);

            if ($property->default !== null && !$exists) {
                data_set($params, $property->name, $property->default);
                $value = $property->default;
                $exists = true;
            }

            if (RpcDefinition::getInstance()->inStructCache($property->type)) {
                $this->handlerDefaultAndType(RpcDefinition::getInstance()->getStruct($property->type), $value);

                data_set($params, $property->name, $value);
                continue;
            }

            if ($this->checkType && $exists) {
                // 检查可空的类型
                if ($value === null && $property->nullable) {
                    continue;
                }

                if (!Language::tryAutoFixType($value, $property->type)) {
                    $prefix = str_replace(sprintf("%s.%s", $this->refInstance->getDeclaringClass()->getName(), $this->name), "", $requestType->namespace);
                    $key = ltrim($prefix ? "$prefix.$property->name" : $property->name, '.');
                    throw new RpcRuntimeException("key `{$key}` is not a {$property->type} instance", 422);
                }

                data_set($params, $property->name, $value);
            }
        }
    }

    private function keyExists ($target, $key) {
        $value = data_get($target, $key, Language::$placeholder);

        return [$value !== Language::$placeholder, $value];
    }

    public function buildMethodParams (RpcContext $context, array $params) {
        $methodParams = [];

        foreach ($this->parameters as $name => $type) {
            if ($type === RpcContext::class) {
                $methodParams[] = $context;
            } else {
                if (is_subclass_of($type, RpcObject::class)) {
                    /** @var RpcObject $object */
                    $object = new $type;

                    RpcDefinition::getInstance()->getMapper()->map($params, $object);
                    $methodParams[] = $object;
                } else {
                    $methodParams[] = $params[$name] ?? null;
                }
            }
        }

        return $methodParams;
    }

    public function loadFromRpcApi (RpcApi $rpcApi) {
        $url = $rpcApi->url;
        if ($rpcApi->addPrefix && config('rpc-server.url.enable_prefix')) {
            $url = path_join(config('rpc-server.url.prefix'), $url);
        }

        $this->httpApi = new HttpApi();
        $this->httpApi->method = $rpcApi->method;
        $this->httpApi->url = $url;
    }

    public function loadFromRpcHeader (RpcHeader $rpcHeader) {
        $header = new Header();
        $header->name = $rpcHeader->name;
        $this->headers[] = $header;
    }

    public function loadFromRpcFormat (RpcFormat $rpcFormat) {
        $this->resSerialize = $rpcFormat->format;
    }

    public function loadFromRpcResponseHeader (RpcResponseHeader $rpcResponseHeader) {
        $header = new ResponseHeader();
        $header->name = $rpcResponseHeader->name;
        $header->value = $rpcResponseHeader->value;
        $this->resHeaders[] = $header;
    }

    public function loadFromRpcAuthority (RpcAuthority $rpcAuthority) {
        $this->needAuthorities[] = $rpcAuthority->need;
    }

    public function loadFromRpcMessageHook (RpcMessageHook $rpcMessageHook) {
        // 兼容旧websocket申明
        if (strpos($rpcMessageHook->name, "ws.") === 0) {
            print "\n###!!!!! RpcMessageHook ws.* for websocket is DEPRECATED and will be removed in a future version. Use RpcWebsocket instead.\n";
            $this->wsNeedConnectMessage = true;
            $this->wsNeedDisconnectMessage = true;
            $this->wsRoom = substr($rpcMessageHook->name, 3);
        } else {
            $this->messageHookName = $rpcMessageHook->name;
        }
    }

    public function loadFromRpcWebsocket (RpcWebsocket $annotation) {
        $this->wsNeedConnectMessage = $annotation->needConnectMessage;
        $this->wsNeedDisconnectMessage = $annotation->needDisconnectMessage;
        $this->wsRoom = $annotation->roomName;
    }

    public function loadFromRpcDisableTypeValidation (RpcDisableTypeValidation $annotation) {
        $this->checkType = $annotation->check_type;
        $this->checkValidation = $annotation->check_validation;
    }

    public function loadFromRpcCron (RpcCron $annotation, $classAliasName) {
        $annotation->every and $this->cronSpec = '@every '.$annotation->every;
        $annotation->entry and $this->cronSpec = '@'.$annotation->entry;
        $annotation->spec and $this->cronSpec = str_replace('\\', '/', $annotation->spec);

        $this->cronHookName = "{$classAliasName}::{$this->name}";
    }

    public function loadFromRpcParam (RpcParam $rpcParam) {
        if ($this->requestType == RpcDefinition::EmptyType) {
            $this->requestType = sprintf("%s.%s", $this->refInstance->getDeclaringClass()->getName(), $this->name);
        }

        $struct = RpcDefinition::getInstance()->notDefinitionStruct($this->requestType);
        $property = new Property();

        $names = explode('.', $rpcParam->name);
        $prefixType = $this->requestType;
        foreach ($names as $index => $child) {
            if ($index == count($names) - 1) {
                $property->require = $rpcParam->require;
                $property->validation = $rpcParam->validation;
                if (is_subclass_of($rpcParam->type, RpcObject::class)) {
                    $_ = RpcDefinition::getInstance()->objectStruct($rpcParam->type);
                }

                $property->type = $rpcParam->type;
                $property->nullable = $rpcParam->nullable;
                $property->name = $child;

                if ($rpcParam->default !== null) {
                    $property->default = $rpcParam->default;
                }
                $property->comment = $rpcParam->comment;
                $struct->properties[] = $property;
            } else {
                $childProperty = $this->findProperty($struct->properties, $child);
                if (!$childProperty) {
                    $childStruct = RpcDefinition::getInstance()->notDefinitionStruct($prefixType.'.'.$child);
                    $childProperty = new Property();
                    $childProperty->name = $child;
                    $childProperty->type = $prefixType.'.'.$child;
                    $struct->properties[] = $childProperty;
                    $struct = $childStruct;
                } else {
                    $struct = RpcDefinition::getInstance()->notDefinitionStruct($childProperty->type);
                }

                $prefixType = $prefixType.'.'.$child;
            }
        }
    }

    private function findProperty ($properties, $name): ?Property {
        /** @var array<Property> $properties */
        foreach ($properties as $property) {
            if ($property->name == $name) {
                return $property;
            }
        }

        return null;
    }

    /**
     * @return array
     * @throws ReflectionException
     */
    public function getParameters (): array {
        if ($this->parameters === null) {
            $this->parameters = [];

            foreach ($this->refInstance->getParameters() as $parameter) {
                if (isset($this->getRequestDeclareMap()[$parameter->name])) {
                    /** @var Property $property */
                    $property = $this->getRequestDeclareMap()[$parameter->name];
                    if ($parameter->isDefaultValueAvailable()) {
                        $property->default = $parameter->getDefaultValue();
                    } else {
                        $property->require = true;
                    }
                }

                if (!isset($this->getRequestDeclareMap()[$parameter->name])
                    && $parameter->getType()
                    && !is_subclass_of($parameter->getType()->getName(), RpcObject::class)
                    && !$parameter->getType()->getName() == RpcContext::class
                ) {
                    throw new RuntimeException("the rpc param [{$parameter->name}] not defined in [{$this->refInstance->class}::{$this->name}]");
                }

                if ($parameter->getType() && is_subclass_of($parameter->getType()->getName(), RpcObject::class)) {
                    if ($this->requestType != RpcDefinition::EmptyType) {
                        if (!is_subclass_of($this->requestType, RpcObject::class)) {
                            throw new RuntimeException("There can only be one in RpcObject and RpcParam");
                        }

                        throw new RuntimeException("RpcObject only support one");
                    }

                    $_ = RpcDefinition::getInstance()->objectStruct($parameter->getType()->getName());

                    $this->requestType = $parameter->getType()->getName();
                }


                $this->parameters[$parameter->name] = optional($parameter->getType())->getName();
            }
        }

        return $this->parameters;
    }

    /**
     * @return array
     */
    public function getRequestDeclareMap (): array {
        if ($this->requestDeclareMap === null) {
            $this->requestDeclareMap = [];
            foreach (RpcDefinition::getInstance()->getStruct($this->requestType)->properties as $property) {
                $this->requestDeclareMap[$property->name] = $property;
            }
        }

        return $this->requestDeclareMap;
    }

    /**
     * @return array
     */
    public function getRequestDeclareNames (): array {
        if ($this->requestDeclareNames === null) {
            $this->requestDeclareNames = [];
            foreach ($this->getRequestDeclareMap() as $name => $property) {
                $this->getPropertyNames($this->requestDeclareNames, "", $property);
            }
        }

        return $this->requestDeclareNames;
    }

    private function getPropertyNames (&$names, $prefix, Property $property) {
        $name = $prefix ? $prefix.'.'.$property->name : $property->name;

        if (RpcDefinition::getInstance()->inStructCache($property->type)) {
            foreach (RpcDefinition::getInstance()->getStruct($property->type)->properties as $childProperty) {
                $this->getPropertyNames($names, $name, $childProperty);
            }
        } else {
            $names[] = $name;
        }
    }

    /**
     * @return ReflectionMethod
     */
    public function getRefInstance (): ReflectionMethod {
        return $this->refInstance;
    }

    public function loadFromAlias (Alias $annotation) {
        $this->alias = $annotation->name;
    }

    public function loadDocument (string $docComment) {
        $this->description = "";

        foreach (explode("\n", $docComment) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;

                $this->description .= $_doc . "\n";
                continue;
            }
            if (strpos($line, '@deprecated') !== false) {
                $this->deprecated = true;
            }
        }
    }
}