<?php


namespace Mainto\RpcServer\RpcServer\Middleware\Method;


use Closure;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\Middleware\MethodMiddleware;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcSession;
use Mainto\RpcServer\Util\Language;

class DataFillAndTypeCheckMiddleware implements MethodMiddleware {

    private static ?DataFillAndTypeCheckMiddleware $instance = null;

    private function __construct () { }

    public static function getInstance (): ?DataFillAndTypeCheckMiddleware {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    public function handle (RpcContext $context, Closure $next, $args = []) {
        $requestStruct = RpcDefinition::getInstance()->getStruct($context->getMethod()->requestType);

        $params = $context->getParams();

        $this->dataFillAndTypeCheck($requestStruct, $params, $context, $context->getMethod()->checkType);

        $context->setParams($params);

        return $next($context);
    }

    private function fromSession(RpcSession $session, &$params, Property $property): array {
        // 保证声明了fromSession的属性，只能从session或者默认值获取
        if ($session && $session->has($property->fromSession)) {
            data_set($params, $property->name, $session->get($property->fromSession));
            $value = $session->get($property->fromSession);
        } else if ($property->defaultValueAvailable){
            data_set($params, $property->name, $property->default);
            $value = $property->default;
        } else {
            throw new RpcRuntimeException("miss session value: {$property->fromSession}", 422);
        }

        return [true, $value];
    }

    private function fromHeader(array $headers, &$params, Property $property): array {
        // 保证声明了fromHeader的属性，只能从session或者默认值获取
        if ($headers && isset($headers[$property->fromHeader])) {
            data_set($params, $property->name, $headers[$property->fromHeader]);
            $value = $headers[$property->fromHeader];
        } else if ($property->defaultValueAvailable){
            data_set($params, $property->name, $property->default);
            $value = $property->default;
        } else {
            throw new RpcRuntimeException("miss header value: {$property->fromHeader}", 422);
        }

        return [true, $value];
    }

    private function dataFillAndTypeCheck (Struct $requestType, array &$params, RpcContext $context, $checkType) {
        $session = $context->getSession();
        $headers = $context->getRequestHeaders();
        foreach ($requestType->properties as $property) {
            if ($property->fromSession) {
                [$exists, $value] = $this->fromSession($session, $params, $property);
            } elseif ($property->fromHeader) {
                [$exists, $value] = $this->fromHeader($headers, $params, $property);
            } else {
                [$exists, $value] = $this->keyExists($params, $property->name);
                if ($property->defaultValueAvailable && !$exists) {
                    data_set($params, $property->name, $property->default);
                    $value = $property->default;
                    $exists = true;
                }
            }

            if ($exists) {
                // 检查可空的类型
                if ($value === null && $property->nullable) {
                    continue;
                }

                if (Str::endsWith($property->type, '[]') && RpcDefinition::getInstance()->inStructCache(substr($property->type, 0, -2))) {
                    foreach ($value as &$child) {
                        $this->dataFillAndTypeCheck(RpcDefinition::getInstance()->getStruct(substr($property->type, 0, -2)), $child, $context, $checkType);
                    }

                    data_set($params, $property->name, $value);
                    continue;
                }

                if (RpcDefinition::getInstance()->inStructCache($property->type)) {
                    $this->dataFillAndTypeCheck(RpcDefinition::getInstance()->getStruct($property->type), $value, $context, $checkType);

                    data_set($params, $property->name, $value);
                    continue;
                }

                if ($checkType) {
                    if (!Language::tryAutoFixType($value, $property->type)) {
                        $prefix = ($requestType instanceof NotDefinitionStruct) ? $requestType->originName : "";
                        $key = ltrim($prefix ? "$prefix.$property->name" : $property->name, '.');
                        throw new RpcRuntimeException("key `{$key}` is not a {$property->type} instance", 422);
                    }

                    data_set($params, $property->name, $value);
                }
            }
        }
    }

    private function keyExists ($target, $key): array {
        $value = data_get($target, $key, Language::$placeholder);

        return [$value !== Language::$placeholder, $value];
    }
}