<?php

namespace Mainto\RpcServer\RpcServer;

use Doctrine\Common\Annotations\AnnotationReader;
use InvalidArgumentException;
use Mainto\RpcServer\Base\Controller;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcAnnotations\RpcWebsocket;
use Mainto\RpcServer\RpcServer\Definition\Definition;
use Mainto\RpcServer\RpcServer\Definition\Method;
use Mainto\RpcServer\Service\RpcParameter;
use Mainto\RpcServer\Service\Struct\Config;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class RpcRouter {

    /**
     * @var RpcRouter
     */
    private static ?RpcRouter $instance = null;

    /**
     * 注解解释器
     * @var AnnotationReader
     */
    private AnnotationReader $_annotationReader;

    /**
     * @var array [classKey => [classInstance => 'ReflectionClass', methodItems => [methodName => ['methodInstance' => 'ReflectionMethod', parameter => 'RpcParameter']]]]
     */
    private array $_classInstancesMap = [];

    /**
     * alias Map
     * @var array ['alias' => 'className']
     */
    private array $aliasMap = [];

    /**
     * @var array ['alias' => controller]
     */
    private array $controllerMap = [];

    /**
     * @var array ['controllerName' => 'aliasName']
     */
    private array $controllerAliasNameMap = [];

    /**
     * @var array ['alias_methodName' => method]
     */
    private array $controllerMethodMap = [];

    /**
     * RpcRouter constructor.
     */
    private function __construct () {
        $this->_annotationReader = new AnnotationReader();
    }

    public static function getInstance () {
        if (self::$instance == null) {
            self::$instance = new RpcRouter();
        }

        return self::$instance;
    }

    /**
     * 注册一个RPC路由
     *
     * @param string $class
     * @param string|null $alias
     */
    public function register (string $class, string $alias = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$alias) {
                $alias = $refClass->getName();
                $pos = strrpos($alias, 'Controllers\\');
                if ($pos !== false) {
                    $alias = substr($alias, $pos + 12);
                } else {
                    throw new RuntimeException($class.' not has alias and path error');
                }
            }

            $this->aliasMap[$class] = $alias;
            $this->addClass($refClass, $alias);
            $this->addController($refClass, $alias);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    /**
     * 为路由增加一个 Class 的所有公共的普通方法
     *
     * @param ReflectionClass $class
     * @param string $alias
     * @throws ReflectionException
     */
    private function addClass (ReflectionClass $class, string $alias) {
        $instance = $this->_classInstancesMap[$alias] ?? null;
        if ($instance) {
            if ($class->isInstance($instance["classInstance"])) {
                return;
            } else {
                $instanceClassName = (new ReflectionClass($instance))->getName();
                throw new InvalidArgumentException("class {$class->getName()} is used by {$instanceClassName}");
            }
        } else {
            $instance = $class->newInstance();
            if ($instance instanceof Controller) {
                $this->_classInstancesMap[$alias] = [
                    'classInstance' => $instance,
                    'methodItems'   => [],
                ];
            } else {
                throw new RuntimeException("register router {$class->getName()} is not support");
            }
        }

        $methods = $class->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod($method, $alias);
        }
    }

    /**
     * 为路由增加一个方法
     *
     * @param ReflectionMethod $method
     * @param string $classAlias
     * @throws ReflectionException
     */
    private function addMethod (ReflectionMethod $method, string $classAlias) {
        if (!$method->isStatic() && $method->isPublic()) {
            $methodName = $method->getShortName();

            if ($methodName[0] == "_") {
                return;
            }

            // register
            $rpcParameter = new RpcParameter();
            $defaultResponse = new Response();
            $defaultResponse->useReturnOKType();
            $defaultResponse->setBody(Body::newJsonBody("{}"));
            $this->_classInstancesMap[$classAlias]['methodItems'][$methodName] = [
                'methodInstance'  => $method,
                'parameter'       => $rpcParameter,
                'defaultResponse' => $defaultResponse,
            ];

            // 对参数反射进行注
            $parameters = $method->getParameters();

            foreach ($parameters as $key => $parameter) {
                $rpcParameter->addParameter($parameter);
            }

            // 注册 Annotation
            $annotations = $this->_annotationReader->getMethodAnnotations($method);
            foreach ($annotations as $annotation) {
                if ($annotation instanceof RpcParam) {
                    if ($annotation->require) {
                        $rpcParameter->setRequired($annotation->name);
                    }

                    if ($annotation->validation) {
                        $rpcParameter->setValidation($annotation->name, $annotation->validation);
                    }

                    if ($annotation->type) {
                        $rpcParameter->setType($annotation->name, $annotation->type);
                    }

                    if ($annotation->default !== null) {
                        $rpcParameter->setDefault($annotation->name, $annotation->default);
                    }

                    if ($annotation->nullable) {
                        $rpcParameter->setNullable($annotation->name);
                    }

                    $rpcParameter->addComment($annotation->name, $annotation->comment);
                    $rpcParameter->addDeclareName($annotation->name);
                } elseif ($annotation instanceof RpcFormat) {
                    switch ($annotation->format) {
                        case "Jsonp":
                        case "Json":
                            // default is json/jsonp
                            break;
                        case "Raw":
                            $defaultResponse->setBody(Body::newRawBody(""));
                            break;
                        default:
                            throw new RuntimeException("format not support: ".$annotation->format);
                    }

                } elseif ($annotation instanceof RpcDisableTypeValidation) {
                    if (!$annotation->check_validation) {
                        $rpcParameter->disableValidationCheck();
                    }

                    if (!$annotation->check_type) {
                        $rpcParameter->disableTypeCheck();
                    }
                }
            }
        }
    }

    public function addController (ReflectionClass $refClass, string $alias) {
        $definition = Definition::getInstance();

        if (isset($this->controllerMap[$alias]) && !$refClass->isInstance($this->controllerMap[$alias]->instance)) {
            throw new InvalidArgumentException(" duplicate class name/alias: {$alias}");
        }

        if (!ends_with($alias, 'Controller')) {
            throw new RuntimeException("class name must be use Controller end");
        }

        $definition->controllers[] = $controller = new \Mainto\RpcServer\RpcServer\Definition\Controller();
        $controller->alias = $alias;
        $controller->shortName = substr($alias, 0, -10);;
        $controller->name = $refClass->getName();
        $controller->instance = $refClass->newInstance();

        if (!($controller->instance instanceof \Mainto\RpcServer\Base\Controller)) {
            throw new RuntimeException("register router {$refClass->getName()} is not support");
        }

        $methods = $refClass->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod2($controller, $method);
        }

        // cache for fast find
        $this->controllerMap[$alias] = $controller;
        $this->controllerAliasNameMap[$alias] = $refClass->getName();
    }

    public function addMethod2(\Mainto\RpcServer\RpcServer\Definition\Controller $controller, ReflectionMethod $refMethod) {
        $methodName = $refMethod->getShortName();
        if ($methodName[0] == "_") {
            return;
        }

        $controller->methods[] = $method = new Method();
        $method->name = $methodName;
        $method->refInstance = $refMethod;

        $defaultResponse = new Response();
        $defaultResponse->useReturnOKType();
        $defaultResponse->setBody(Body::newJsonBody("{}"));

        $annotations = app(AnnotationReader::class)->getMethodAnnotations($refMethod);
        foreach ($annotations as $annotation) {
            if ($annotation instanceof RpcApi) {
                $method->loadFromRpcApi($annotation);
            } elseif ($annotation instanceof RpcHeader) {
                $method->loadFromRpcHeader($annotation);
            } elseif ($annotation instanceof RpcFormat) {
                $method->loadFromRpcFormat($annotation);
            } elseif ($annotation instanceof RpcResponseHeader) {
                $method->loadFromRpcResponseHeader($annotation);
            } elseif ($annotation instanceof RpcAuthority) {
                $method->loadFromRpcAuthority($annotation);
            } elseif ($annotation instanceof RpcMessageHook) {
                $method->loadFromRpcMessageHook($annotation);
            } elseif ($annotation instanceof RpcWebsocket) {
                $method->loadFromRpcWebsocket($annotation);
            } elseif ($annotation instanceof RpcDisableTypeValidation) {
                $method->loadFromRpcDisableTypeValidation($annotation);
            } elseif ($annotation instanceof RpcCron) {
                $method->loadFromRpcCron($annotation, $controller->alias);
            } elseif ($annotation instanceof RpcParam) {
                $method->loadFromRpcParam($annotation);
            }
        }

        $this->controllerMethodMap[$controller->alias][$method->name] = $method;
    }

    public function getController (string $fullClassName): ?\Mainto\RpcServer\RpcServer\Definition\Controller {
        return $this->controllerMap[$fullClassName] ?? null;
    }

    public function getControllerMethod (string $fullClassName, $methodName): ?\Mainto\RpcServer\RpcServer\Definition\Method {
        return $this->controllerMethodMap[$fullClassName][$methodName] ?? null;
    }

    public function getConfig (): Config {
        return Config::createFromDefinition(Definition::getInstance());
    }

    /**
     * 获取控制器名称
     *
     * @param $class
     * @return string|null
     */
    public function getControllerAlias ($class) {
        return $this->aliasMap[$class] ?? null;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public function getBaseRouterConfig () {
        $struct = Config::createFromDefinition(Definition::getInstance());

        $config = [
            'name'       => config('rpc-server.service_name'),
            'paths'      => $struct->getPaths()->toArray(),
            'crons'      => [],
            'messages'   => [],
            'websockets' => $struct->getWebsockets()->toArray(),
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = $struct->getCrons()->toArray();
        }

        if (config('rpc-server.sidecar.rabbitmq.enable')) {
            $config['messages'] = $struct->getMessages()->toArray();
        }

        return $config;
    }

    /**
     * @return array
     */
    public function getClassInstancesMap (): array {
        return $this->_classInstancesMap;
    }

    /**
     * @param $className
     * @param $method
     * @return array|null ['methodInstance' => 'ReflectionMethod', parameter => 'RpcParameter']
     */
    public function getMethodItemByClassNameAndMethodName ($className, $method) {
        return $this->_classInstancesMap[$className]['methodItems'][$method] ?? null;
    }
}