<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\RpcServer\Exceptions\RpcMethodNotFoundException;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\BodyType;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\Service\RpcParameter;

class RpcInvoke {
    /**
     * @var array [RpcFormat => Response]
     */
    public static array $cacheResponse = [];

    public static function initCacheResponse() {
        $jsonResponse = new Response();
        $jsonResponse->useReturnOKType();
        $jsonResponse->setBody(Body::newJsonBody("{}"));

        self::$cacheResponse['Json'] = $jsonResponse;
        self::$cacheResponse['Jsonp'] = $jsonResponse;

        $rawResponse = new Response();
        $rawResponse->useReturnOKType();
        $rawResponse->setBody(Body::newRawBody(""));
        self::$cacheResponse['Raw'] = $rawResponse;
    }

    /**
     * @param Request $request
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    public static function invokeBase (Request $request) {
        $fullClassName = $request->getCallClassName().'Controller';

        $controller = RpcRouter::getInstance()->getController($fullClassName);
        if (!$controller) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $fullClassName, $request->getCallMethodName()));
        }

        $method = RpcRouter::getInstance()->getControllerMethod($fullClassName, $request->getCallMethodName());

        if (!$method) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $fullClassName, $request->getCallMethodName()));
        }

        // todo 使用新的结构解析参数
        /** @var RpcParameter $parameter */
        ['parameter' => $parameter] = RpcRouter::getInstance()->getClassInstancesMap()[$fullClassName]['methodItems'][$request->getCallMethodName()];

        $rpcParamsService = new RpcParamsService($parameter, $request->getAllParams());
        $context = new RpcContext($request, $rpcParamsService->getRpcParams());
        $baseRet = $method->refInstance->invokeArgs($controller->instance, $rpcParamsService->buildMethodParam($context));

        // session
        $context->saveSession();

        return [$context, $baseRet, self::$cacheResponse[$method->resSerialize]];
    }

    public static function invoke (Request $request): Response {
        /** @var RpcContext $context */
        /** @var Response $response */
        [$context, $ret, $response] = self::invokeBase($request);

        if ($ret instanceof Response) {
            return $ret;
        }

        // body
        $body = $response->getBody();
        switch ($body->getBodyType()) {
            case BodyType::JSON:
                if ($ret instanceof Jsonable) {
                    $body->setBodyBytes($ret->toJson());
                } elseif ($ret instanceof Arrayable) {
                    $body->setBodyBytes(json_encode($ret->toArray()));
                } else {
                    $body->setBodyBytes(json_encode($ret));
                }
                break;
            case BodyType::RAW:
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RpcNotSupportException("body type not support: ".$body->getBodyType());
        }

        // responseHeaders
        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        return $response;
    }
}