<?php

namespace Mainto\RpcServer\RpcUtil\Tool\MonitorDriver;

use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\RpcClient\RpcClient;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcServer\RpcContext;
use RuntimeException;
use Throwable;

class SidecarPrometheus extends MonitorDriveAbstract {
    private static ?RpcClient $rpcClient = null;

    private static ?Request $cacheRequest = null;

    private static array $actionMap = [
        'set' => ['type'=> 'gauge', 'action'=> 'Set'],
        'add' => ['type'=> 'gauge', 'action'=> 'Add'],
        'sub' => ['type'=> 'gauge', 'action'=> 'Sub'],
        'inc' => ['type'=> 'gauge', 'action'=> 'Inc'],
        'dec' => ['type'=> 'gauge', 'action'=> 'Dec'],
        'summary' => ['type'=> 'summary', 'action'=> 'Observe'],
    ];

    public function __construct () {
        parent::__construct();
        if (self::$cacheRequest === null) {
            $requestExtendHeader = new RequestExtendHeader();
            self::$cacheRequest = new Request();
            self::$cacheRequest->setType(RequestHeaderType::SidecarMonitorType);
            self::$cacheRequest->setExtend($requestExtendHeader);
        }
    }

    public function throwable ($request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $traceStr = "\n";
        foreach (explode("\n", $e->getTraceAsString()) as $key => $value) {
            $traceStr .= $value."\n";
            if (strpos($value, 'php-rpc-framework/src/RpcServer/RpcInvoke.php') !== false
                && strpos($value, 'ReflectionMethod->invokeArgs()') !== false) {
                $traceStr .= "[framework trace]";
                break;
            }
        }

        $this->send($source, [
            "request"       => $this->transObject($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => $traceStr,
        ]);
    }

    /**
     * request throwable error log
     *
     * @param $request
     * @param $source
     * @param Throwable $e
     */
    public function requestThrowable (Request $request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $traceStr = "\n";
        foreach (explode("\n", $e->getTraceAsString()) as $key => $value) {
            $traceStr .= $value."\n";
            if (strpos($value, 'php-rpc-framework/src/RpcServer/RpcInvoke.php') !== false
                && strpos($value, 'ReflectionMethod->invokeArgs()') !== false) {
                $traceStr .= "[framework trace]";
                break;
            }
        }

        $this->send($source, [
            "header"        => $request->getAllHeaders(),
            "request"       => $request->getAllParams(),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => $traceStr,
        ]);
    }

    /**
     * send message to sidecar sls service
     *
     * @param $source
     * @param $logStore
     * @param $contents
     */
    private function send (string $source, array $contents) {
        try {
            /** @var RequestExtendHeader $ext */
            $ext = self::$cacheRequest->getExt();
            $contents['service_name'] = config('rpc-server.service_name');
            self::$cacheRequest->setBody(Body::newJsonBody(json_encode($contents)));

            if (self::$rpcClient === null) {
                self::$rpcClient = RpcClient::getMonitorClient();
            }
            self::$cacheRequest->setTraceId(RpcContext::$currentTraceId);
            $response = self::$rpcClient->Do(self::$cacheRequest);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $e) {
            (optional(self::$rpcClient))->close();
            self::$rpcClient = null;
            Log::error($e, $contents);
            throw new RuntimeException("can not find this monitor server", 0, $e);
        }
    }

    private function push (array $opt) {
        $this->send($this->getTraceInfo()['source'], $opt);
    }


    private function dealParam (string $function, string $name, float $value = null) {
        return [
            "type"   => data_get(self::$actionMap, 'type'),
            "name"   => $name,
            "content" => [
                'action' => data_get(self::$actionMap, 'action'),
                'value' => $value
            ],
        ];
    }

    public function set (string $name, float $value) {
        $this->push($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function add (string $name, float $value) {
        $this->push($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function sub (string $name, float $value) {
        $this->push($this->dealParam(__FUNCTION__, $name, $value));
    }

    public function inc (string $name) {
        $this->push($this->dealParam(__FUNCTION__, $name));
    }

    public function dec (string $name) {
        $this->push($this->dealParam(__FUNCTION__, $name));
    }

    public function summary (string $name, float $value) {
        $this->push($this->dealParam(__FUNCTION__, $name, $value));
    }
}
