<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Service\RpcParameter;
use stdClass;

class RpcParamsService {
    /**
     * @var RpcParameter
     */
    private RpcParameter $parameter;

    /**
     * @var array
     */
    private array $params;

    /**
     * 临时变量，用于 data_get($this->params, $key, $temp) 比较
     * @var stdClass
     */
    private stdClass $_temp;

    public function __construct (RpcParameter $parameter, array $params) {
        $this->_temp = new stdClass();
        $this->parameter = $parameter;
        $this->params = $params;

        // fill default
        foreach ($this->parameter->getDefaultMap() as $key => $value) {
            [$exists, $_] = $this->keyExists($this->params, $key);
            if (!$exists) {
                data_set($this->params, $key, $value);
            }
        }

        $this->paramsTypeValidate();
        $this->paramsDataValidate();
    }

    private function keyExists($target, $key) {
        $value = data_get($target, $key, $this->_temp);

        return [$value !== $this->_temp, $value];
    }

    private function paramsTypeValidate (): void {
        if ($this->parameter->isEnableTypeCheck()) {
            foreach ($this->parameter->getTypeMap() as $key => $type) {
                [$exists, $value] = $this->keyExists($this->params, $key);
                if ($exists) {
                    // 检查可空的类型
                    if ($value === null && ($this->parameter->getNullableMap()[$key] ?? false)) {
                        continue;
                    }

                    if (!$this->tryAutoFixType($value, $type)) {
                        throw new RpcRuntimeException("key `{$key}` is not a {$type} instance", 422);
                    }
                    data_set($this->params, $key, $value);
                }
            }
        }
    }

    /**
     * @throws ValidationException
     */
    private function paramsDataValidate (): void {
        $validationMap = $this->parameter->getValidationMap();
        foreach ($this->parameter->getMustHave() as $item) {
            if (!isset($validationMap[$item])) {
                $validationMap[$item] = "required";
            } elseif (strpos($validationMap[$item], "required") === false) {
                $validationMap[$item] .= "|required";
            }
        }

        if ($this->parameter->isEnableValidationCheck() && $validationMap) {

            $validator = Validator::make($this->params, $validationMap, [], []);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
        }
    }

    /**
     * 检测类型
     *
     * @param $value
     * @param $type
     * @return bool
     */
    private function tryAutoFixType (&$value, $type): bool {
        switch ($type) {
            case "string":
                if (is_string($value)) {
                    $value = strval($value);
                    return true;
                }
                return false;
            case "uint":
                if (is_numeric($value) && $value >= 0) {
                    $value = intval($value);
                    return true;
                }
                return false;
            case "int":
                if (is_numeric($value)) {
                    $value = intval($value);
                    return true;
                }
                return false;
            case "double":
                if (is_numeric($value)) {
                    $value = floatval($value);
                    return true;
                }
                return false;
            case "unsignedDouble":
                if (is_numeric($value) && $value >= 0) {
                    $value = floatval($value);
                    return true;
                }
                return false;
            case "bool":
                if (is_bool($value)) {
                    return true;
                } else {
                    if (is_string($value) || is_numeric($value)) {
                        switch (strtolower($value)) {
                            case 'true':
                            case 1:
                                $value = true;
                                return true;
                            case 'false':
                            case 0:
                                $value = false;
                                return true;
                        }
                    }
                    return false;
                }
            case "array":
                return is_array($value);
            case "array<int>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (int)$val;
                    }
                    return true;
                }
                return false;
            case  "array<uint>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $val = (int)$val;
                        if ($val < 0) return false;

                        $value[$key] = (int)$val;
                    }
                    return true;
                }
                return false;
            case  "array<string>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (string)$val;
                    }
                    return true;
                }
                return false;
            case "mixed":
                return true;
        }

        return false;
    }

    /**
     * 构造方法参数列表
     * @param RpcContext $context
     * @return array
     */
    public function buildMethodParam (RpcContext $context): array {
        $methodParams = [];
        foreach ($this->parameter->getMethodParameters() as $key => $parameter) {

            if ($this->parameter->getContextPos() === $key) {
                // 兼容模式
                if ($parameter->getType()->getName() !== RpcContext::class) {
                    $ctxClass = $parameter->getType()->getName();
                    $methodParams[] = new $ctxClass($context->getRequest(), $context->getRpcParams());
                } else {
                    $methodParams[] = $context;
                }
            } else {
                $name = $parameter->getName();
                $methodParams[] = $this->params[$name] ?? null;
            }
        }

        return $methodParams;
    }

    public function getRpcParams (): RpcParams {
        $rpcParams = new RpcParams($this->params);

        $rpcParams->_declareNames = $this->parameter->getDeclareName();

        return $rpcParams;
    }
}
