<?php

namespace Mainto\MRPC\Service;

use Doctrine\Common\Annotations\AnnotationReader;
use InvalidArgumentException;
use Mainto\MRPC\Base\Controller;
use Mainto\MRPC\Annotations\RpcApi;
use Mainto\MRPC\Annotations\RpcAuthority;
use Mainto\MRPC\Annotations\RpcCron;
use Mainto\MRPC\Annotations\RpcDisableTypeValidation;
use Mainto\MRPC\Annotations\RpcFormat;
use Mainto\MRPC\Annotations\RpcHeader;
use Mainto\MRPC\Annotations\RpcMessageHook;
use Mainto\MRPC\Annotations\RpcParam;
use Mainto\MRPC\Annotations\RpcResponseHeader;
use Mainto\MRPC\Protocol\Request\RpcParameter;
use Mainto\MRPC\Service\Struct\Config;
use Mainto\MRPC\Service\Struct\Cron;
use Mainto\MRPC\Service\Struct\Message;
use Mainto\MRPC\Service\Struct\Path;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class RpcRouter {

    /**
     * @var RpcRouter
     */
    private static ?RpcRouter $instance = null;

    /**
     * url前缀
     * @var string
     */
    private $urlPrefix;

    /**
     * 注解解释器
     * @var AnnotationReader
     */
    private AnnotationReader $_annotationReader;

    /**
     * @var array
     */
    private array $_classInstancesMap = [];

    /**
     * @var array
     */
    private array $_paramsMap = [];

    /**
     * service config
     * @var Config
     */
    private Config $config;

    /**
     * RpcRouter constructor.
     */
    private function __construct () {
        $this->urlPrefix = config('rpc-server.url.prefix');

        $this->_annotationReader = new AnnotationReader();

        $this->config = new Config();
    }

    public static function getInstance () {
        if (self::$instance == null) {
            self::$instance = new RpcRouter();
        }

        return self::$instance;
    }

    /**
     * 注册一个RPC路由
     *
     * @param string $class
     * @param string|null $alias
     */
    public function register (string $class, string $alias = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$alias) {
                $alias = $refClass->getName();
                $pos = strrpos($alias, 'Controllers\\');
                if ($pos !== false) {
                    $alias = substr($alias, $pos + 12);
                } else {
                    throw new RuntimeException($class.'not has alias and path error');
                }
            }

            $this->addClass($refClass, $alias);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    public function getConfig(): Config {
        return $this->config;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public function getBaseRouterConfig () {
        $config = [
            'name'     => config('rpc-server.service_name'),
            'paths'    => $this->config->getPaths()->toArray(),
            'crons'    => [],
            'messages' => [],
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = $this->config->getCrons()->toArray();
        }

        if (config('rpc-server.rabbit_mq.enable')) {
            $config['messages'] = $this->config->getMessages()->toArray();
        }

        return $config;
    }

    /**
     * 为路由增加一个 Class 的所有公共的普通方法
     *
     * @param ReflectionClass $class
     * @param string $alias
     * @throws ReflectionException
     */
    private function addClass (ReflectionClass $class, string $alias) {
        $instance = $this->classesMap[$alias] ?? null;
        if ($instance) {
            if ($class->isInstance($instance)) {
                return;
            } else {
                $instanceClassName = (new ReflectionClass($instance))->getName();
                throw new InvalidArgumentException("class {$class->getName()} is used by {$instanceClassName}");
            }
        } else {
            $instance = $class->newInstance();
            if ($instance instanceof Controller) {
                $this->_classInstancesMap[$alias] = $instance;
            } else {
                throw new RuntimeException("register router {$class->getName()} is not support");
            }
        }

        $methods = $class->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod($class, $method, $alias);
        }
    }

    /**
     * 为路由增加一个方法
     *
     * @param ReflectionClass $class
     * @param ReflectionMethod $method
     * @param string $classAlias
     * @throws ReflectionException
     */
    private function addMethod (ReflectionClass $class, ReflectionMethod $method, string $classAlias) {
        if (!$method->isStatic() && $method->isPublic()) {
            $className = $classAlias;
            $methodName = $method->getShortName();
            $classMethodMapKey = "{$className}::{$methodName}";

            if ($methodName[0] == "_") {
                return;
            }

            // 对参数反射进行注
            $parameters = $method->getParameters();

            $rpcParameter = new RpcParameter();
            foreach ($parameters as $key => $parameter) {
                $rpcParameter->addParameter($parameter);
            }

            // 注册 Annotation
            $annotations = $this->_annotationReader->getMethodAnnotations($method);
            $pathItem = new Path();
            $cronItem = new Cron();
            $messageItem = new Message();
            foreach ($annotations as $annotation) {
                if ($annotation instanceof RpcApi) {
                    $url = $annotation->url;
                    if ($annotation->addPrefix && config('rpc-server.url.enable_prefix')) {
                        $url = sprintf("%s/%s", $this->urlPrefix, ltrim($url, '/'));
                    }

                    $pathItem->setClass($className);
                    $pathItem->setMethod($methodName);
                    $pathItem->setHttpMethod($annotation->method);
                    $pathItem->setHttpUrl($url);
                } elseif ($annotation instanceof RpcParam) {
                    if ($annotation->require) {
                        $rpcParameter->setRequired($annotation->name);
                    }

                    if ($annotation->validation) {
                        $rpcParameter->setValidation($annotation->name, $annotation->validation);
                    }

                    if ($annotation->type) {
                        $rpcParameter->setType($annotation->name, $annotation->type);
                    }

                    if ($annotation->default) {
                        $rpcParameter->setDefault($annotation->name, $annotation->default);
                    }

                    if ($annotation->nullable) {
                        $rpcParameter->setNullable($annotation->name);
                    }

                    $rpcParameter->addComment($annotation->name, $annotation->comment);
                    $rpcParameter->addDeclareName($annotation->name);
                } elseif ($annotation instanceof RpcHeader) {
                    $pathItem->addHeader($annotation->name);
                } elseif ($annotation instanceof RpcFormat) {
                    $pathItem->setFormat($annotation->format);
                } elseif ($annotation instanceof RpcResponseHeader) {
                    $pathItem->addResponseHeader($annotation->name, $annotation->value);
                } elseif ($annotation instanceof RpcAuthority) {
                    $pathItem->addAuthority($annotation->need);
                } elseif ($annotation instanceof RpcMessageHook) {
                    $messageItem->setClass($className);
                    $messageItem->setMethod($methodName);
                    $messageItem->setMessageHookName($annotation->name);

                } elseif ($annotation instanceof RpcDisableTypeValidation) {
                    if (!$annotation->check_validation) {
                        $rpcParameter->disableValidationCheck();
                    }

                    if (!$annotation->check_type) {
                        $rpcParameter->disableTypeCheck();
                    }
                } elseif ($annotation instanceof RpcCron) {
                    $cronItem->setClass($className);
                    $cronItem->setMethod($methodName);
                    $cronItem->setSpec($annotation->spec, $annotation->every, $annotation->entry);
                    $cronItem->setCronHookName($classMethodMapKey);
                    $this->config->addCron($cronItem);
                }
            }

            $pathItem->addResponseHeader('X-Process', 'backend/'.config('rpc-server.service_name')."/1");
            if ($pathItem->getHttpMethod()) {
                $this->config->addPath($pathItem);
            }

            if (!$messageItem->isEmpty()) {
                $this->config->addMessage($messageItem);
            }

            if (!$rpcParameter->isEmpty()) {
                $this->_paramsMap[$classMethodMapKey] = $rpcParameter;
            }
        }
    }

    /**
     * @return array
     */
    public function getClassInstancesMap (): array {
        return $this->_classInstancesMap;
    }
}