<?php


namespace Mainto\MRPC\Protocol;


use Exception;
use Illuminate\Support\Facades\Log;
use Mainto\MRPC\Protocol\Common\Types\RequestHeaderType;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Request\RequestReader;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Tool\Socket;

abstract class RpcConnection {
    /**
     * @var string
     */
    protected string $streamId = "";

    /**
     * @var string
     */
    protected string $server;

    /**
     * @var string
     */
    protected string $port;

    /**
     * 啰嗦模式
     * @var bool
     */
    public static $verboseMode = false;

    protected ?Response $defaultResponse = null;

    /**
     * RpcServer constructor.
     * @param string $server
     * @param string $port
     * @param string $streamId
     */
    protected function __construct (string $server, string $port, string $streamId) {
        self::$verboseMode = env('RPC_VERBOSE_MODE', null);

        if (self::$verboseMode === null) {
            self::$verboseMode = env('APP_DEBUG', false);
        }

        $this->server = $server;
        $this->port = $port;
        $this->streamId = $streamId;
    }

    /**
     * RpcServer destruct
     */
    public function __destruct () {
        Log::error("server exit...");
    }

    abstract public function onRequest(RequestReader $request): Response;

    abstract public function onClose(Socket $socket);

    public function start() {
        $socket = new Socket($this->server, $this->port);
        $socket->send(static::initRequest()->encode());

        while (true) {
            try {
                $request = RequestReader::ReadRequestFrom($socket);
                $this->dumpRequestReader($request);

                switch ($request->getType()) {
                    case RequestHeaderType::SystemCheckType:
                        $response = $this->serviceDefaultResponse();
                        break;
                    default:
                        $response = $this->onRequest($request);
                }

                $this->dumpResponse($response);
            } catch (Exception $exception) {
                Log::error($this->streamId);
                Log::error($exception);
                break;
            } finally {

            }
        }

        $this->onClose($socket);

        $socket->close();
    }

    abstract public static function initRequest(): Request;

    public function serviceDefaultResponse() {
        if ($this->defaultResponse === null) {
            $this->defaultResponse = new Response();
            $this->defaultResponse->useReturnOKType();
        }

        return $this->defaultResponse;
    }

    protected function dumpRequestReader(RequestReader $request) {
        Log::debug("[{$this->streamId}] req: ".$request->toJson(JSON_UNESCAPED_UNICODE));
    }

    protected function dumpResponse(Response $response) {
        Log::debug("[{$this->streamId}] res: ".$response->toJson(JSON_UNESCAPED_UNICODE));
    }
}