<?php


namespace Mainto\MRPC\Protocol\Request;


use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\Bytes\Bytes;
use Mainto\MRPC\Tool\IO\IOUtil;
use Mainto\MRPC\Tool\IO\ReadCloser;
use RuntimeException;

class RequestReader implements Jsonable {
    private ReadCloser $reader;

    private int $magicNum;
    private bool $hasBody;
    private int $type;
    private bool $isStream;

    private string $traceId;
    private string $callClassName;
    private string $callMethodName;
    private bool $readBody;
    private ?Body $body = null;

    private function __construct (ReadCloser $reader) {
        $this->reader = $reader;
    }

    public static function ReadRequestFrom (ReadCloser $reader) {
        $r = new self($reader);

        $r->init();

        return $r;
    }

    private function init () {
        $_ = $this->readerHeader();

        $this->readBody = !$this->hasBody;
    }

    private function readerHeader () {
        $headerSize = $this->readFull(2);

        $size = Binary::$littleEndian::strToUint16($headerSize) - 2;

        if ($size > 542) { // max size 544 - 2
            throw new RuntimeException("ErrHeaderTooLarge: ".$size." ".bin2hex($headerSize));
        }

        $headerByte = $this->readFull($size);

        if (!Bytes::equal(substr($headerByte, 0, 2), Version::ProtocolVersion)) {
            throw new RuntimeException("ErrProtocolVersion");
        }

        $this->magicNum = Binary::$littleEndian::strToUint16(substr($headerByte, 2, 2));
        $this->hasBody = $headerByte[4] === "\x01";

        $this->type = Binary::$littleEndian::strToUint16(substr($headerByte, 13, 2)) & BaseType::TypeMask;

        $this->isStream = ($this->type & BaseType::TypeMask) !== "\x00";

        $this->traceId = substr($headerByte, 15, 16);

        $classLen = Binary::strToUint8($headerByte[31]);
        $this->callClassName = substr($headerByte, 32, $classLen);

        $methodLen = Binary::strToUint8($headerByte[32 + $classLen]);

        $this->callMethodName = substr($headerByte, 32 + $classLen + 1, $methodLen);

        return $size + 2;
    }

    private function readFull ($size) {
        return IOUtil::readFullSize($this->reader, $size);
    }

    public function getBody (): ?Body {
        if (!$this->hasBody) {
            return null;
        }

        $this->readBody = true;

        if ($this->body !== null) {
            return $this->body;
        }

        return $this->body = Body::newBodyFromReader($this->reader);
    }

    public function getExtend (): ?Extend {
        if ($this->magicNum === 0) {
            return null;
        }
        if (!$this->readBody) {
            throw new RuntimeException("ErrMustReadBodyFirst");
        }


        $extend = ExtendFactory::getManagedRequestExtendByMagicNum($this->magicNum);
        $extend->ReadFrom($this->reader);

        return $extend;
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->traceId;
    }

    /**
     * @return string
     */
    public function getCallClassName (): string {
        return $this->callClassName;
    }

    /**
     * @return string
     */
    public function getCallMethodName (): string {
        return $this->callMethodName;
    }

    /**
     * @return int
     */
    public function getType (): int {
        return $this->type;
    }

    /**
     * @return ReadCloser
     */
    public function getReader (): ReadCloser {
        return $this->reader;
    }

    public function toJson ($options = 0) {
        return json_encode([
            "type"            => BaseType::Map[$this->type],
            "traceId"        => $this->getTraceId(),
            "callClassName"  => $this->getCallClassName(),
            "callMethodName" => $this->getCallMethodName(),
            "body"           => optional($this->getBody())->toJson($options),
//            "ext"           => $this->getExtend()->toJson($options),
        ], $options);
    }
}