<?php


namespace Mainto\MRPC\Protocol\Common;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mainto\MRPC\Protocol\Common\Types\BodyType;
use Mainto\MRPC\Util\Binary;
use Mainto\MRPC\Util\Bytes\Buffer;
use Mainto\MRPC\Util\Bytes\Bytes;
use Mainto\MRPC\Util\IO\IO;
use Mainto\MRPC\Util\IO\Reader;
use Mainto\MRPC\Util\IO\ReaderFrom;
use Mainto\MRPC\Util\IO\Writer;
use Mainto\MRPC\Util\IO\WriterTo;
use Mainto\MRPC\Util\Reader\LimitReader;
use RuntimeException;

class Body implements WriterTo, ReaderFrom {
    /**
     * uint8
     * @var int
     */
    private int $bodyType;

    /**
     * uint32
     * @var string
     */
    private string $bodySize;


    private string $body = "";

    private Buffer $bodyReader;

    /**
     * Body constructor.
     */
    private function __construct () {
    }

    public static function newBodyFromReader (Reader $reader): Body {
        $object = new self();
        $object->readFrom($reader);

        return $object;
    }

    public function readFrom (Reader $reader): int {
        $read = $reader->read(5);
        $this->bodyType = Binary::strToUint8($read[0]);
        $this->bodySize = Binary::$littleEndian::strToUint32(substr($read, 1, 4));

        $buffer = new Buffer();
        IO::copy($buffer, IO::limitReader($reader, $this->bodySize));

        $this->bodyReader = $buffer;

        return 5;
    }

    public static function newRawBody (string $body): Body {
        $object = new self();
        $object->bodyType = BodyType::RAW;
        $object->bodySize = strlen($body);
        $object->body = $body;

        $buffer = new Buffer();
        $buffer->writeString($body);

        $object->bodyReader = $buffer;

        return $object;
    }

    public function setBodyBytes(string $body) {
        $this->bodySize = strlen($body);
        $this->body = $body;

        $this->bodyReader->reset();
        $this->bodyReader->writeString($body);
    }

    public static function newJsonBody (string $body): Body {
        $object = new self();
        $object->bodyType = BodyType::JSON;
        $object->bodySize = strlen($body);
        $object->body = $body;

        $buffer = new Buffer();
        $buffer->writeString($body);

        $object->bodyReader = $buffer;

        return $object;
    }

    public function reset() {
        $this->bodySize = 0;
        $this->body = "";
        $this->bodyReader->reset();
    }

    public function cacheBody () {
        $buffer = Bytes::newBuffer();

        IO::copy($buffer, $this->getBodyReader());

        $this->bodyReader = $buffer;

        return $this;
    }

    public function getBodyReader () {
        return IO::limitReader($this->bodyReader, $this->bodySize);
    }

    public function getBodyBytes(bool $cache = false): string {
        $buffer = Bytes::newBuffer();

        IO::copy($buffer, $this->getBodyReader());

        if ($cache) {
            $this->bodyReader = $buffer;
        }

        return $buffer->bytes();
    }

    public function encode () {
        return IO::IOWriterToBytes($this);
    }

    public function getContent() {
        switch ($this->bodyType) {
            case BodyType::JSON:
                $decode = json_decode($this->getBodyBytes(true), true);
                if ($decode === false) {
                    Log::error("body decode fail");
                }

                return $decode;
            case BodyType::RAW:
                return $this->getBodyBytes(true);
            default:
                throw new RuntimeException("body type not support");
        }
    }

    public function toRaw(): string {
        switch ($this->bodyType) {
            case BodyType::RAW:
                return $this->getBodyBytes(true);
            case BodyType::JSON:
            default:
                throw new RuntimeException("body type not support");
        }
    }

    private function encodeSize (): string {
        $makeBytes = Binary::makeBytes(5);
        $makeBytes[0] = Binary::intToUint8($this->bodyType);

        Binary::$littleEndian::putUint32($makeBytes, 1, $this->bodySize);

        return $makeBytes;
    }

    public function writeTo (Writer $w): int {
        return $w->write($this->encodeSize()) + IO::copy($w, $this->getBodyReader());
    }

    public function checkParams () {
        if ($this->bodySize > 10 * 1024 * 1024) {
            throw new RuntimeException("body package is too large");
        }
    }

    /**
     * @return int
     */
    public function getBodyType (): int {
        return $this->bodyType;
    }

    public function checkLimitReadOver(): bool {
        if ($this->bodyReader instanceof LimitReader) {
            return $this->bodyReader->n <= 0;
        }

        return true;
    }

    public function dumpToWriter(Writer $writer) {
        $writer->write(sprintf("%12s: %d(%s)\n", "body type", $this->bodyType, BodyType::Map[$this->bodyType]));
        $writer->write(sprintf("%12s: %d\n", "body size", $this->bodySize));

        if ($this->bodyType == BodyType::JSON || $this->bodyType == BodyType::RAW) {
            $writer->write(sprintf("%12s:\n", "body content"));
            $writer->write($this->getBodyBytes(true));
        }
    }
}