<?php


namespace Mainto\MRPC\Server;


use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Mainto\MRPC\Exceptions\RpcRuntimeException;
use Mainto\MRPC\Service\RpcParameter;

class RpcParamsService {
    /**
     * @var RpcParameter
     */
    private RpcParameter $parameter;

    /**
     * @var array
     */
    private array $params;

    public function __construct (RpcParameter $parameter, array $params) {
        $this->parameter = $parameter;
        $this->params = $params;


        $this->paramsTypeValidate();
        $this->paramsDataValidate();
    }

    private function paramsTypeValidate (): void {
        if ($this->parameter->isEnableTypeCheck()) {
            foreach ($this->parameter->getTypeMap() as $key => $type) {
                if (isset($this->params[$key])) {
                    if (!$this->checkType($key, $type)) {
                        throw new RpcRuntimeException("key `{$key}` is not a {$type} instance", 422);
                    }
                }
            }
        }
    }

    /**
     * @throws ValidationException
     */
    private function paramsDataValidate (): void {
        $validationMap = $this->parameter->getValidationMap();
        foreach ($this->parameter->getMustHave() as $item) {
            if (!isset($validationMap[$item])) {
                $validationMap[$item] = "required";
            } elseif (strpos($validationMap[$item], "required") === false) {
                $validationMap[$item] .= "|required";
            }
        }

        if ($this->parameter->isEnableValidationCheck() && $validationMap) {

            $validator = Validator::make($this->params, $validationMap, [], []);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
        }
    }

    /**
     * 检测类型
     *
     * @param $name
     * @param $type
     * @return bool
     */
    private function checkType ($name, $type): bool {
        $value = $this->params[$name] ?? null;

        // 检查可空的类型
        if ($value === null && ($this->parameter->getNullableMap()[$name] ?? false)) {
            return true;
        }

        switch ($type) {
            case "string":
                if (is_string($value)) {
                    $this->params[$name] = strval($value);
                    return true;
                }
                return false;
            case "uint":
                if (is_numeric($value) && $value >= 0) {
                    $this->params[$name] = intval($value);
                    return true;
                }
                return false;
            case "int":
                if (is_numeric($value)) {
                    $this->params[$name] = intval($value);
                    return true;
                }
                return false;
            case "double":
                if (is_numeric($value)) {
                    $this->params[$name] = floatval($value);
                    return true;
                }
                return false;
            case "unsignedDouble":
                if (is_numeric($value) && $value >= 0) {
                    $this->params[$name] = floatval($value);
                    return true;
                }
                return false;
            case "bool":
                if (is_bool($value)) {
                    return true;
                } else {
                    if (is_string($value) || is_numeric($value)) {
                        switch (strtolower($value)) {
                            case 'true':
                            case 1:
                                $this->params[$name] = true;
                                return true;
                            case 'false':
                            case 0:
                                $this->params[$name] = false;
                                return true;
                        }
                    }
                    return false;
                }
            case "array":
                return is_array($value);
            case "array<int>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (int)$val;
                    }
                    $this->params[$name] = $value;
                    return true;
                }
                return false;
            case  "array<uint>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $val = (int)$val;
                        if ($val < 0) return false;

                        $value[$key] = (int)$val;
                    }
                    $this->params[$name] = $value;
                    return true;
                }
                return false;
            case  "array<string>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (string)$val;
                    }
                    $this->params[$name] = $value;
                    return true;
                }
                return false;
            case "mixed":
                return true;
        }

        return false;
    }

    /**
     * 构造方法参数列表
     * @param RpcContext $context
     * @return array
     */
    public function buildMethodParam (RpcContext $context): array {
        $methodParams = [];
        foreach ($this->parameter->getMethodParameters() as $key => $parameter) {

            if ($this->parameter->getContextPos() === $key) {
                // 兼容模式
                if ($parameter->getType()->getName() !== RpcContext::class) {
                    $ctxClass = $parameter->getType()->getName();
                    $methodParams[] = new $ctxClass($context->getRequest(), $context->getRpcParams());
                } else {
                    $methodParams[] = $context;
                }
            } else {
                $name = $parameter->getName();

                if (isset($this->params[$name])) {
                    $methodParams[] = $this->params[$name] ?? null;
                } elseif (isset($this->parameter->getDefaultMap()[$name])) {
                    $methodParams[] = $this->parameter->getDefaultMap()[$name];
                }
            }
        }

        return $methodParams;
    }

    public function getRpcParams (): RpcParams {
        $rpcParams = new RpcParams($this->params);

        $rpcParams->_declareNames = $this->parameter->getDeclareName();

        return $rpcParams;
    }
}