<?php


namespace Mainto\MRPC\Server;


use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\MRPC\Exceptions\BaseNotReportServiceException;
use Mainto\MRPC\Log\RpcLog;
use Mainto\MRPC\Protocol\Common\Types\RequestHeaderType;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Request\RequestReader;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendError;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Util\Net\Conn;
use Throwable;

abstract class BaseRpcConnection {
    protected Conn $conn;

    public bool $verboseMode;

    /**
     * @var ConnectionHandler
     */
    private ConnectionHandler $handler;
    private string $streamId;

    private int $pid;

    protected Response $cacheErrResponse;

    private bool $isStopped = false;

    public function __construct (Conn $conn, string $streamId, ConnectionHandler $handler) {
        $this->verboseMode = config('rpc-server.verbose_mode');

        $this->handler = $handler;
        $this->streamId = $streamId;
        $this->conn = $conn;
        $this->pid = getmypid();

        $this->cacheErrResponse = new Response();
        $this->cacheErrResponse->useReturnErrType();
        $this->cacheErrResponse->setExtend(new ResponseExtendError());
    }

    /**
     * 处理请求 && 友好的处理异常
     *
     * @param Request $request
     * @return Response
     */
    private function handleRequest (Request $request): Response {
        try {
            return $this->handler->onRequest($request);
        } catch (Throwable $exception) {
            /** @var ResponseExtendError $errExt */
            $errExt = $this->cacheErrResponse->getExt();
            $errExt->setTraceId(isset($request) ? $request->getTraceId() : "");
            $errExt->setErrMsg($exception->getMessage());

            if ($exception instanceof ValidationException) {
                $errExt->setErrorCode(422);
            } elseif ($exception instanceof InvalidArgumentException) {
                $errExt->setErrorCode(500);
            } else {
                if (!($exception instanceof BaseNotReportServiceException)) {
                    RpcLog::getInstance()->logRequestThrowable($request, $exception, "{$request->getCallClassName()}::{$request->getCallMethodName()}");
                }

                $errExt->setErrorCode(intval($exception->getCode()) ?: 500);
            }

            if ($exception instanceof Exception) {
                app(ExceptionHandler::class)->report($exception);
            } else {
                Log::error($exception, ["streamId" => $this->streamId]);
            }
            return $this->cacheErrResponse;
        } finally {
            $this->resourceRecover();
        }
    }

    /**
     * 回收可能导致有问题的资源
     */
    private function resourceRecover () {
        while (DB::transactionLevel()) {
            DB::rollBack();
        }
    }

    public function start () {
        $this->handler->beforeStart($this->conn);
        while (!$this->isStopped) {
            try {
                $request = RequestReader::ReadRequestFrom($this->conn)->toRequest();
            } catch (Throwable $exception) {
                Log::error($exception, ["streamId" => $this->streamId, 'pid' => $this->pid]);
                break;
            }

            $this->dumpRequest($request);
            $startTime = microtime(true);

            switch ($request->getType()) {
                case RequestHeaderType::SidecarStopType:
                    Log::debug(sprintf("[%s %d] exit... by recv SidecarStopType signal", $this->streamId, $this->pid));
                    $response = new Response();
                    $response->useReturnOKType();
                    $this->isStopped = true;
                    break;
                default:
                    $response = $this->handleRequest($request);
            }

            $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));
            $this->dumpResponse($response);
            $response->writeTo($this->conn);
            $response->resetBodyAndExt();
        }

        $this->close();
    }

    public function close () {
        $this->handler->onClose($this->conn);

        $this->conn->close();
    }

    protected function dumpRequest (Request $request) {
        if ($this->verboseMode) {
            Log::debug("[server recv req {$this->pid} {$this->streamId}] \n".$request->dump());
        }
    }

    protected function dumpResponse (Response $response) {
        if ($this->verboseMode) {
            Log::debug("[server send res {$this->pid} {$this->streamId}] \n".$response->dump());
        }
    }
}