<?php

namespace Mainto\MRPC\Log\LogDriver;

use Illuminate\Support\Facades\Log;
use Mainto\MRPC\Client\BaseRpcClient;
use Mainto\MRPC\Client\RpcUnixClient;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Types\RequestHeaderType;
use Mainto\MRPC\Protocol\Common\Types\ResponseHeaderType;
use Mainto\MRPC\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendError;
use RuntimeException;
use Throwable;

class SidecarSLS extends LogDriveAbstract {
    private static ?BaseRpcClient $slsClient;

    private static ?Request $cacheRequest;

    public function __construct () {
        parent::__construct();

        if ($this->localMode) {
            throw new RuntimeException('agent sls log driver can not support local mode');
        }

        if (self::$cacheRequest === null) {
            $requestExtendHeader = new RequestExtendHeader();
            self::$cacheRequest = new Request();
            self::$cacheRequest->setType(RequestHeaderType::SidecarSLSType);
            self::$cacheRequest->setExtend($requestExtendHeader);
        }
    }

    /**
     * log throwable error log
     *
     * @param $request
     * @param $source
     * @param Throwable $e
     */
    public function logRequestThrowable (Request $request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "request"       => $this->transObject($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => "\n".$e->getTraceAsString(),
        ]);
    }


    /**
     * send message to sidecar sls service
     *
     * @param $source
     * @param $logStore
     * @param $contents
     */
    private function send (string $source, string $logStore, array $contents) {
        try {
            /** @var RequestExtendHeader $ext */
            $ext = self::$cacheRequest->getExt();
            $ext->setHeader("channel", [$logStore]);

            $contents['method'] = $source;
            self::$cacheRequest->setBody(Body::newJsonBody(json_encode($contents)));

            if (self::$slsClient === null) {
                self::$slsClient = RpcUnixClient::getSLSClient();
            }

            $response = self::$slsClient->Do(self::$cacheRequest);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $e) {
            (optional(self::$slsClient))->close();
            self::$slsClient = null;
            Log::error($e, $contents);
            throw new RuntimeException("can not find this sls server", 0, $e);
        }
    }

    /**
     * log info message
     *
     * @param $title
     * @param $message
     */
    public function info (string $title, $message) {
        $this->log("INFO", $title, $message);
    }

    /**
     * log some message
     *
     * @param string $level
     * @param string $title
     * @param $message
     */
    protected function log (string $level, string $title, $message) {
        $traceInfo = $this->getTraceInfo();
        $this->send($traceInfo['source'], "log", [
            "offset"  => $traceInfo['file'],
            "level"   => $level,
            "title"   => $title,
            "content" => $this->transObject($message),
        ]);
    }

    /**
     * log debug message
     *
     * @param $title
     * @param $message
     */
    public function debug (string $title, $message) {
        $this->log("DEBU", $title, $message);
    }

    /**
     * log notice message
     *
     * @param $title
     * @param $message
     */
    public function notice (string $title, $message) {
        $this->log("NOTI", $title, $message);
    }

    /**
     * log warning message
     *
     * @param $title
     * @param $message
     */
    public function warning (string $title, $message) {
        $this->log("WARN", $title, $message);
    }

    /**
     * log error message
     *
     * @param $source
     * @param $title
     * @param $message
     */
    public function error (string $title, $message) {
        $this->log("ERRO", $title, $message);
    }

    /**
     * log critical message
     *
     * @param $title
     * @param $message
     */
    public function critical (string $title, $message) {
        $this->log("CRIT", $title, $message);
    }
}