<?php


namespace Mainto\MRPC\Protocol\Common;


use Mainto\MRPC\Protocol\Common\Types\BodyType;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\Bytes\Bytes;
use Mainto\MRPC\Tool\IO\IO;
use Mainto\MRPC\Tool\IO\Reader;
use Mainto\MRPC\Tool\IO\ReaderFrom;
use Mainto\MRPC\Tool\IO\Writer;
use Mainto\MRPC\Tool\IO\WriterTo;
use RuntimeException;

class Body implements WriterTo, ReaderFrom {
    /**
     * uint8
     * @var int
     */
    private int $bodyType;

    /**
     * uint32
     * @var string
     */
    private string $bodySize;


    private string $body = "";

    private Reader $bodyReader;

    /**
     * Body constructor.
     */
    private function __construct () {
    }

    public static function newBodyFromReader (Reader $reader): Body {
        $object = new self();
        $object->readFrom($reader);

        return $object;
    }

    public function readFrom (Reader $reader): int {
        $read = $reader->read(5);
        $this->bodyType = Binary::strToUint8($read[0]);
        $this->bodySize = Binary::$littleEndian::strToUint32(substr($read, 1, 4));
        $this->bodyReader = $reader;

        return 5;
    }

    public static function newJsonBody (string $body): Body {
        $object = new self();
        $object->bodyType = BodyType::JSON;
        $object->bodySize = strlen($body);
        $object->body = $body;
        $object->bodyReader = Bytes::newReader($body);

        return $object;
    }

    public function cacheBody () {
        $buffer = Bytes::newBuffer();

        IO::copy($buffer, $this->getBodyReader());

        $this->bodyReader = $buffer;
    }

    public function getBodyReader () {
        return IO::limitReader($this->bodyReader, $this->bodySize);
    }

    public function encode () {
        return IO::IOWriterToBytes($this);
    }

    private function encodeSize (): string {
        $makeBytes = Binary::makeBytes(5);
        $makeBytes[0] = Binary::intToUint8($this->bodyType);

        Binary::$littleEndian::putUint32($makeBytes, 1, $this->bodySize);

        return $makeBytes;
    }

    public function writeTo (Writer $w): int {
        return $w->write($this->encodeSize()) + IO::copy($w, $this->getBodyReader());
    }

    public function checkParams () {
        if ($this->bodySize > 10 * 1024 * 1024) {
            throw new RuntimeException("body package is too large");
        }
    }
}