<?php


namespace Mainto\MRPC\Protocol\Request;


use Mainto\MRPC\Exceptions\RpcRuntimeException;
use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Types\RequestHeaderType;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\IO\IO;
use Mainto\MRPC\Tool\IO\Writer;
use Mainto\MRPC\Tool\IO\WriterTo;

class Request implements WriterTo {
    private string $traceId;
    private string $callClassName;
    private string $callMethodName;
    private string $type;
    private ?Extend $ext = null;
    private ?Body $body;
    private bool $isStream = false;

    /**
     * @param string $traceId
     */
    public function setTraceId (string $traceId): void {
        $this->traceId = $traceId;
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->traceId;
    }

    private function encodeHeaderLen () {
        if (strlen($this->callClassName) > 255 || strlen($this->callMethodName) > 255) {
            throw new RpcRuntimeException("header too large.");
        }

        $size = 0;
        $size += 2;  // head size
        $size += 2;  // version
        $size += 2;  // extend magic number
        $size += 1;  // has body
        $size += 8;  // 预留
        $size += 2;  // stream | empty  | main type | second type
        $size += 16; // request trace id

        $size += 1; // class size
        $size += strlen($this->callClassName);
        $size += 1; // method size
        $size += strlen($this->callMethodName);

        // uint16
        return $size;
    }

    private function encodeHeader () {
        $size = $this->encodeHeaderLen();
        $buf = Binary::makeBytes($size);
        Binary::$littleEndian::putUint16($buf, 0, $size);

        $buf[2] = Version::ProtocolVersion[0];
        $buf[3] = Version::ProtocolVersion[1];
        if ($this->ext !== null) {
            Binary::$littleEndian::putUint16($buf, 4, $this->ext->magicNum());
        }

        if ($this->body !== null) {
            $buf[6] = "\x01";
        }

        // type

        if ($this->isStream) {
            $this->type = $this->type | BaseType::StreamOpen | $this->type;
        }

        Binary::$littleEndian::putUint16($buf, 15, $this->type);

        Binary::copy($buf, 17, 33, $this->traceId);

        $classLen = strlen($this->callClassName);
        Binary::putUint8($buf, 33, $classLen);
        Binary::copy($buf, 34, 34 + $classLen, $this->callClassName);

        $methodLen = strlen($this->callMethodName);
        Binary::putUint8($buf, 34 + $classLen, $methodLen);
        Binary::copy($buf, 34 + $classLen + 1, 34 + $classLen + 1 + $methodLen, $this->callMethodName);

        return $buf;
    }

    public function encode() {
        return IO::IOWriterToBytes($this);
    }

    public function writeTo (Writer $w): int {
        $b = $this->encodeHeader();
        $totalWrite = $w->write($b);

	    if ($this->body !== null) {
            $totalWrite += $this->body->writeTo($w);
        }

	    if ($this->ext) {
            $totalWrite += $this->ext->writeTo($w);
        }

	    return $totalWrite;
    }

    /**
     * @param string $class
     * @param string $method
     */
    public function setCallClassAndMethod (string $class, string $method): void {
        $this->callClassName = $class;
        $this->callMethodName = $method;
    }

    public function useStreamOpenType () {
        $this->type = BaseType::StreamOpen;
        return $this;
    }

    public function useStreamChunkType () {
        $this->type = BaseType::StreamChunk;
        return $this;
    }

    public function useStreamCloseType () {
        $this->type = BaseType::StreamClose;
        return $this;
    }

    public function useStreamErrorCloseType () {
        $this->type = BaseType::StreamErrorClose;
        return $this;
    }

    public function useUrlInvokeGetType () {
        $this->type = RequestHeaderType::UrlInvokeGetType;
        return $this;
    }

    public function useUrlInvokePostType () {
        $this->type = RequestHeaderType::UrlInvokePostType;
        return $this;
    }

    public function useUrlInvokePutType () {
        $this->type = RequestHeaderType::UrlInvokePutType;
        return $this;
    }

    public function useUrlInvokeDeleteType () {
        $this->type = RequestHeaderType::UrlInvokeDeleteType;
        return $this;
    }

    public function useUrlInvokePatchType () {
        $this->type = RequestHeaderType::UrlInvokePatchType;
        return $this;
    }

    public function useInvokeNormalType () {
        $this->type = RequestHeaderType::InvokeNormalType;
        return $this;
    }

    public function useMessageNormalType () {
        $this->type = RequestHeaderType::MessageNormalType;
        return $this;
    }

    public function useMessageDelayType () {
        $this->type = RequestHeaderType::MessageDelayType;
        return $this;
    }

    public function useMessageRetryType () {
        $this->type = RequestHeaderType::MessageRetryType;
        return $this;
    }

    public function useCronNormalType () {
        $this->type = RequestHeaderType::CronNormalType;
        return $this;
    }

    public function useWebsocketNormalType () {
        $this->type = RequestHeaderType::WebsocketNormalType;
        return $this;
    }

    public function useWebsocketConnectType () {
        $this->type = RequestHeaderType::WebsocketConnectType;
        return $this;
    }

    public function useWebsocketDisconnectType () {
        $this->type = RequestHeaderType::WebsocketDisconnectType;
        return $this;
    }

    public function useSystemCheckType () {
        $this->type = RequestHeaderType::SystemCheckType;
        return $this;
    }

    public function setTypeWithExtend (int $getType, ?Extend $extent) {
        $this->type = $getType;
        $this->ext = $extent;
    }

    public function setExtend (?Extend $extent) {
        $this->ext = $extent;
    }

    public function setBody (?Body $body) {
        $this->body = $body;
    }

    /**
     * @return Body|null
     */
    public function getBody (): ?Body {
        return $this->body;
    }

    public function checkParams() {
        if (strlen($this->callClassName) > 255 || strlen($this->callMethodName) > 255) {
            throw new RpcRuntimeException("header too large.");
        }

        if ($this->body !== null) {
            $this->body->checkParams();
        }

        if ($this->ext !== null) {
            $this->ext->checkParams();
        }
    }
}