<?php


namespace Mainto\MRPC\Server;


use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\MRPC\Protocol\Common\Types\RequestHeaderType;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Request\RequestReader;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendError;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Util\Net\Conn;
use Throwable;

abstract class BaseRpcConnection {
    protected Conn $conn;

    /**
     * 啰嗦模式
     * @var bool
     */
    public static $verboseMode = false;

    /**
     * @var ConnectionHandler
     */
    private ConnectionHandler $handler;
    private string $streamId;

    private int $pid;

    protected Response $cacheErrResponse;
    protected ResponseExtendError $cacheErrResponseExt;

    private bool $isStopped = false;

    public function __construct (Conn $conn, string $streamId, ConnectionHandler $handler) {
        self::$verboseMode = env('RPC_VERBOSE_MODE', null);

        if (self::$verboseMode === null) {
            self::$verboseMode = env('APP_DEBUG', false);
        }
        $this->handler = $handler;
        $this->streamId = $streamId;
        $this->conn = $conn;
        $this->pid = getmypid();

        $this->cacheErrResponse = new Response();
        $this->cacheErrResponse->useReturnErrType();
        $this->cacheErrResponseExt = new ResponseExtendError();
    }

    /**
     * 处理请求 && 友好的处理异常
     *
     * @param Request $request
     * @return Response
     */
    private function handleRequest (Request $request): Response {
        try {
            return $this->handler->onRequest($request);
        } catch (Throwable $exception) {
            $this->cacheErrResponseExt->setTraceId(isset($request) ? $request->getTraceId() : "");
            $this->cacheErrResponseExt->setErrMsg($exception->getMessage());

            if ($exception instanceof ValidationException) {
                $this->cacheErrResponseExt->setErrorCode(422);
            } elseif ($exception instanceof InvalidArgumentException) {
                $this->cacheErrResponseExt->setErrorCode(500);
            } else {
                $this->cacheErrResponseExt->setErrorCode(intval($exception->getCode()) ?: 500);
            }
            $this->cacheErrResponse->setExtend($this->cacheErrResponseExt);

            Log::error($exception, ["streamId" => $this->streamId]);
            return $this->cacheErrResponse;
        }
    }

    public function start () {
        $this->handler->beforeStart($this->conn);
        while (!$this->isStopped) {
            try {
                $request = RequestReader::ReadRequestFrom($this->conn)->toRequest();
            } catch (Throwable $exception) {
                Log::error($exception, ["streamId" => $this->streamId, 'pid' => $this->pid]);
                break;
            }

            $this->dumpRequest($request);
            $startTime = microtime(true);

            switch ($request->getType()) {
                case RequestHeaderType::SidecarStopType:
                    Log::debug(sprintf("[%s %d] exit... by recv SidecarStopType signal", $this->streamId, $this->pid));
                    $response = new Response();
                    $response->useReturnOKType();
                    $this->isStopped = true;
                    break;
                default:
                    $response = $this->handleRequest($request);
            }

            $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));
            $this->dumpResponse($response);
            $response->writeTo($this->conn);
            $response->resetBodyAndExt();
        }

        $this->close();
    }

    public function close () {
        $this->handler->onClose($this->conn);

        $this->conn->close();
    }

    protected function dumpRequest (Request $request) {
        if (self::$verboseMode) {
            Log::debug("[{$this->pid} {$this->streamId}] \n".$request->dump());
        }
    }

    protected function dumpResponse (Response $response) {
        if (self::$verboseMode) {
            Log::debug("[{$this->pid} {$this->streamId}] \n".$response->dump());
        }
    }
}