<?php


namespace Mainto\MRPC\Tool;


use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\GuzzleException;

class EtcdClient {
    // KV
    const URI_PUT = 'kv/put';
    const URI_RANGE = 'kv/range';

    // Lease
    const URI_GRANT = 'lease/grant';
    const URI_REVOKE = 'kv/lease/revoke';
    const URI_KEEPALIVE = 'lease/keepalive';

    /**
     * @var string host:port
     */
    protected string $server;
    /**
     * @var string api version
     */
    protected string $version;

    /**
     * @var HttpClient
     */
    protected HttpClient $httpClient;

    /**
     * @var bool 友好输出, 只返回所需字段
     */
    protected bool $pretty = false;

    /**
     * @var string|null auth token
     */
    protected $token = null;

    public function __construct ($server = '127.0.0.1:2379', $version = 'v3alpha') {
        $this->server = rtrim($server);
        if (strpos($this->server, 'http') !== 0) {
            $this->server = 'http://'.$this->server;
        }
        $this->version = trim($version);

        $baseUri = sprintf('%s/%s/', $this->server, $this->version);
        $this->httpClient = new HttpClient(
            [
                'base_uri' => $baseUri,
                'timeout'  => 30,
            ]
        );
    }

    public function setPretty ($enabled) {
        $this->pretty = $enabled;
    }

    /**
     * Put puts the given key into the key-value store.
     * A put request increments the revision of the key-value
     * store\nand generates one event in the event history.
     *
     * @param string $key
     * @param string $value
     * @param array $options 可选参数
     *                       int  lease
     *                       bool   prev_kv
     *                       bool   ignore_value
     *                       bool   ignore_lease
     * @return array|BadResponseException
     */
    public function put ($key, $value, array $options = []) {
        $params = [
            'key'   => $key,
            'value' => $value,
        ];

        $params = $this->encode($params);
        $options = $this->encode($options);
        $body = $this->request(self::URI_PUT, $params, $options);
        $body = $this->decodeBodyForFields(
            $body,
            'prev_kv',
            ['key', 'value',]
        );

        if (isset($body['prev_kv']) && $this->pretty) {
            return $this->convertFields($body['prev_kv']);
        }

        return $body;
    }

    // region kv

    /**
     * string类型key用base64编码
     *
     * @param array $data
     * @return array
     */
    protected function encode (array $data) {

        foreach ($data as $key => $value) {
            if (is_string($value)) {
                $data[$key] = base64_encode($value);
            }
        }

        return $data;
    }

    /**
     * 发送HTTP请求
     *
     * @param string $uri
     * @param array $params  请求参数
     * @param array $options 可选参数
     * @return array|BadResponseException
     * @throws GuzzleException
     */
    protected function request ($uri, array $params = [], array $options = []) {
        if ($options) {
            $params = array_merge($params, $options);
        }
        // 没有参数, 设置一个默认参数
        if (!$params) {
            $params['php-etcd-client'] = 1;
        }
        $data = [
            'json' => $params,
        ];
        if ($this->token) {
            $data['headers'] = ['Grpc-Metadata-Token' => $this->token];
        }

        $response = $this->httpClient->request('post', $uri, $data);
        $content = $response->getBody()->getContents();

        $body = json_decode($content, true);
        if ($this->pretty && isset($body['header'])) {
            unset($body['header']);
        }

        return $body;
    }

    /**
     * 指定字段base64解码
     *
     * @param array $body
     * @param string $bodyKey
     * @param array $fields 需要解码的字段
     * @return array
     */
    protected function decodeBodyForFields (array $body, $bodyKey, array $fields) {
        if (!isset($body[$bodyKey])) {
            return $body;
        }
        $data = $body[$bodyKey];
        if (!isset($data[0])) {
            $data = [$data];
        }
        foreach ($data as $key => $value) {
            foreach ($fields as $field) {
                if (isset($value[$field])) {
                    $data[$key][$field] = base64_decode($value[$field]);
                }
            }
        }

        if (isset($body[$bodyKey][0])) {
            $body[$bodyKey] = $data;
        } else {
            $body[$bodyKey] = $data[0];
        }

        return $body;
    }

    protected function convertFields (array $data) {
        if (!isset($data[0])) {
            return $data['value'];
        }

        $map = [];
        foreach ($data as $index => $value) {
            $key = $value['key'];
            $map[$key] = $value['value'];
        }

        return $map;
    }

    /**
     * Gets the key or a range of keys
     *
     * @param string $key
     * @param array $options
     *         string range_end
     *         int    limit
     *         int    revision
     *         int    sort_order
     *         int    sort_target
     *         bool   serializable
     *         bool   keys_only
     *         bool   count_only
     *         int  min_mod_revision
     *         int  max_mod_revision
     *         int  min_create_revision
     *         int  max_create_revision
     * @return array|BadResponseException
     */
    public function get ($key, array $options = []) {
        $params = [
            'key' => $key,
        ];
        $params = $this->encode($params);
        $options = $this->encode($options);
        $body = $this->request(self::URI_RANGE, $params, $options);
        $body = $this->decodeBodyForFields(
            $body,
            'kvs',
            ['key', 'value',]
        );

        if (isset($body['kvs']) && $this->pretty) {
            return $this->convertFields($body['kvs']);
        }

        return $body;
    }

    // endregion kv

    // region lease

    /**
     * get all keys with prefix
     *
     * @param string $prefix
     * @return array|BadResponseException
     */
    public function getKeysWithPrefix ($prefix) {
        $prefix = trim($prefix);
        if (!$prefix) {
            return [];
        }
        $lastIndex = strlen($prefix) - 1;
        $lastChar = $prefix[$lastIndex];
        $nextAsciiCode = ord($lastChar) + 1;
        $rangeEnd = $prefix;
        $rangeEnd[$lastIndex] = chr($nextAsciiCode);

        return $this->get($prefix, ['range_end' => $rangeEnd]);
    }

    /**
     * LeaseGrant creates a lease which expires if the server does not receive a
     * keepAlive\nwithin a given time to live period. All keys attached to the lease
     * will be expired and\ndeleted if the lease expires.
     * Each expired key generates a delete event in the event history.",
     *
     * @param int $ttl    TTL is the advisory time-to-live in seconds.
     * @param int $id     ID is the requested ID for the lease.
     *                    If ID is set to 0, the lessor chooses an ID.
     * @return array|BadResponseException
     * @throws GuzzleException
     */
    public function grant ($ttl, $id = 0) {
        $params = [
            'TTL' => $ttl,
            'ID'  => $id,
        ];


        return $this->request(self::URI_GRANT, $params);
    }

    // endregion lease

    // region auth

    /**
     * revokes a lease. All keys attached to the lease will expire and be deleted.
     *
     * @param int $id    ID is the lease ID to revoke. When the ID is revoked,
     *                   all associated keys will be deleted.
     * @return array|BadResponseException
     * @throws GuzzleException
     */
    public function revoke ($id) {
        $params = [
            'ID' => $id,
        ];

        return $this->request(self::URI_REVOKE, $params);
    }

    /**
     * keeps the lease alive by streaming keep alive requests
     * from the client\nto the server and streaming keep alive responses
     * from the server to the client.
     *
     * @param int $id ID is the lease ID for the lease to keep alive.
     * @return array|BadResponseException
     * @throws GuzzleException
     */
    public function keepAlive ($id) {
        $params = [
            'ID' => $id,
        ];

        $body = $this->request(self::URI_KEEPALIVE, $params);

        if (!isset($body['result'])) {
            return $body;
        }
        // response "result" field, etcd bug?
        return [
            'ID'  => $body['result']['ID'],
            'TTL' => $body['result']['TTL'],
        ];
    }
}