<?php


namespace Mainto\MRPC\Protocol\Request;


use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\Bytes\Bytes;
use Mainto\MRPC\Tool\IO\Reader;
use RuntimeException;

class RequestReader {
    private Reader $reader;

    private int $magicNum;
    private bool $hasBody;
    private int $type;
    private bool $isStream;

    private string $traceId;
    private string $callClassName;
    private string $callMethodName;
    private bool $readBody;

    private function __construct (Reader $reader) {
        $this->reader = $reader;
    }

    public static function ReadRequestFrom (Reader $reader) {
        $r = new self($reader);

        $r->init();

        return $r;
    }

    private function init () {
        $_ = $this->readerHeader();

        $this->readBody = !$this->hasBody;
    }

    private function readerHeader () {
        $headerSize = $this->readFull(2);

        $size = Binary::$littleEndian::strToUint16($headerSize) - 2;

        if ($size > 542) { // max size 544 - 2
            throw new RuntimeException("ErrHeaderTooLarge");
        }

        $headerByte = $this->readFull($size);

        if (!Bytes::equal(substr($headerByte, 0, 2), Version::ProtocolVersion)) {
            throw new RuntimeException("ErrProtocolVersion");
        }

        $this->magicNum = Binary::$littleEndian::strToUint16(substr($headerByte, 2, 2));
        $this->hasBody = $headerByte[4] === "\x01";

        $this->type = Binary::$littleEndian::strToUint16(substr($headerByte, 13, 2)) & BaseType::TypeMask;

        $this->isStream = ($this->type & BaseType::TypeMask) !== "\x00";

        $this->traceId = substr($headerByte, 15, 16);

        $classLen = Binary::strToUint8($headerByte[31]);
        $this->callClassName = substr($headerByte, 32, $classLen);

        $methodLen = Binary::strToUint8($headerByte[32 + $classLen]);

        $this->callMethodName = substr($headerByte, 32 + $classLen + 1, $methodLen);

        return $size + 2;
    }

    private function readFull ($size) {
        $buf = "";

        $allReader = 0;
        while (true) {
            $read = $this->reader->read($size - $allReader);

            $buf .= $read;
            $allReader += strlen($read);

            if ($allReader >= $size) {
                return $buf;
            }
        }

        throw new RuntimeException("read full err");
    }

    public function getBody () {
        if (!$this->hasBody) {
            return null;
        }

        $this->readBody = true;

        return Body::newBodyFromReader($this->reader);
    }

    public function getExtend (): ?Extend {
        if ($this->magicNum === 0) {
            return null;
        }
        if (!$this->readBody) {
            throw new RuntimeException("ErrMustReadBodyFirst");
        }


        $extend = ExtendFactory::getManagedRequestExtendByMagicNum($this->magicNum);
        $extend->ReadFrom($this->reader);

        return $extend;
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->traceId;
    }

    /**
     * @return string
     */
    public function getCallClassName (): string {
        return $this->callClassName;
    }

    /**
     * @return string
     */
    public function getCallMethodName (): string {
        return $this->callMethodName;
    }

    /**
     * @return int
     */
    public function getType (): int {
        return $this->type;
    }
}