<?php

namespace Mainto\MRPC\Server;

use Doctrine\Common\Annotations\AnnotationReader;
use InvalidArgumentException;
use Mainto\MRPC\Annotations\RpcWebsocket;
use Mainto\MRPC\Base\Controller;
use Mainto\MRPC\Annotations\RpcApi;
use Mainto\MRPC\Annotations\RpcAuthority;
use Mainto\MRPC\Annotations\RpcCron;
use Mainto\MRPC\Annotations\RpcDisableTypeValidation;
use Mainto\MRPC\Annotations\RpcFormat;
use Mainto\MRPC\Annotations\RpcHeader;
use Mainto\MRPC\Annotations\RpcMessageHook;
use Mainto\MRPC\Annotations\RpcParam;
use Mainto\MRPC\Annotations\RpcResponseHeader;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Service\RpcParameter;
use Mainto\MRPC\Service\Struct\Config;
use Mainto\MRPC\Service\Struct\Cron;
use Mainto\MRPC\Service\Struct\Message;
use Mainto\MRPC\Service\Struct\Path;
use Mainto\MRPC\Service\Struct\Websocket;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class RpcRouter {

    /**
     * @var RpcRouter
     */
    private static ?RpcRouter $instance = null;

    /**
     * url前缀
     * @var string
     */
    private $urlPrefix;

    /**
     * 注解解释器
     * @var AnnotationReader
     */
    private AnnotationReader $_annotationReader;

    /**
     * @var array [classKey => [classInstance => 'ReflectionClass', methodItems => [methodName => ['methodInstance' => 'ReflectionMethod', parameter => 'RpcParameter']]]]
     */
    private array $_classInstancesMap = [];

    /**
     * service config
     * @var Config
     */
    private Config $config;

    /**
     * RpcRouter constructor.
     */
    public function __construct () {
        $this->urlPrefix = config('rpc-server.url.prefix');

        $this->_annotationReader = new AnnotationReader();

        $this->config = new Config();
    }

    public static function getInstance () {
        if (self::$instance == null) {
            self::$instance = new RpcRouter();
        }

        return self::$instance;
    }

    /**
     * 注册一个RPC路由
     *
     * @param string $class
     * @param string|null $alias
     */
    public function register (string $class, string $alias = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$alias) {
                $alias = $refClass->getName();
                $pos = strrpos($alias, 'Controllers\\');
                if ($pos !== false) {
                    $alias = substr($alias, $pos + 12);
                } else {
                    throw new RuntimeException($class.'not has alias and path error');
                }
            }

            $this->addClass($refClass, $alias);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    public function getConfig (): Config {
        return $this->config;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public function getBaseRouterConfig () {
        $config = [
            'name'     => config('rpc-server.service_name'),
            'paths'    => $this->config->getPaths()->toArray(),
            'crons'    => [],
            'messages' => [],
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = $this->config->getCrons()->toArray();
        }

        if (config('rpc-server.sidecar.rabbitmq.enable')) {
            $config['messages'] = $this->config->getMessages()->toArray();
        }

        return $config;
    }

    /**
     * 为路由增加一个 Class 的所有公共的普通方法
     *
     * @param ReflectionClass $class
     * @param string $alias
     * @throws ReflectionException
     */
    private function addClass (ReflectionClass $class, string $alias) {
        $instance = $this->_classInstancesMap[$alias] ?? null;
        if ($instance) {
            if ($class->isInstance($instance["classInstance"])) {
                return;
            } else {
                $instanceClassName = (new ReflectionClass($instance))->getName();
                throw new InvalidArgumentException("class {$class->getName()} is used by {$instanceClassName}");
            }
        } else {
            $instance = $class->newInstance();
            if ($instance instanceof Controller) {
                $this->_classInstancesMap[$alias] = [
                    'classInstance' => $instance,
                    'methodItems'   => [],
                ];
            } else {
                throw new RuntimeException("register router {$class->getName()} is not support");
            }
        }

        $methods = $class->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod($method, $alias);
        }
    }

    /**
     * 为路由增加一个方法
     *
     * @param ReflectionMethod $method
     * @param string $classAlias
     * @throws ReflectionException
     */
    private function addMethod (ReflectionMethod $method, string $classAlias) {
        if (!$method->isStatic() && $method->isPublic()) {
            $className = $classAlias;
            $methodName = $method->getShortName();
            $classMethodMapKey = "{$className}::{$methodName}";

            if ($methodName[0] == "_") {
                return;
            }

            // register
            $rpcParameter = new RpcParameter();
            $defaultResponse = new Response();
            $defaultResponse->useReturnOKType();
            $defaultResponse->setBody(Body::newJsonBody("{}"));
            $this->_classInstancesMap[$classAlias]['methodItems'][$methodName] = [
                'methodInstance'  => $method,
                'parameter'       => $rpcParameter,
                'defaultResponse' => $defaultResponse,
            ];

            // 对参数反射进行注
            $parameters = $method->getParameters();

            foreach ($parameters as $key => $parameter) {
                $rpcParameter->addParameter($parameter);
            }

            // 注册 Annotation
            $annotations = $this->_annotationReader->getMethodAnnotations($method);
            $pathItem = new Path();
            $cronItem = new Cron();
            $messageItem = new Message();
            $websocket = new Websocket();
            foreach ($annotations as $annotation) {
                if ($annotation instanceof RpcApi) {
                    $url = $annotation->url;
                    if ($annotation->addPrefix && config('rpc-server.url.enable_prefix')) {
                        $url = path_join($this->urlPrefix, $url);
                    }

                    $pathItem->setClass($className);
                    $pathItem->setMethod($methodName);
                    $pathItem->setHttpMethod($annotation->method);
                    $pathItem->setHttpUrl($url);
                } elseif ($annotation instanceof RpcParam) {
                    if ($annotation->require) {
                        $rpcParameter->setRequired($annotation->name);
                    }

                    if ($annotation->validation) {
                        $rpcParameter->setValidation($annotation->name, $annotation->validation);
                    }

                    if ($annotation->type) {
                        $rpcParameter->setType($annotation->name, $annotation->type);
                    }

                    if ($annotation->default) {
                        $rpcParameter->setDefault($annotation->name, $annotation->default);
                    }

                    if ($annotation->nullable) {
                        $rpcParameter->setNullable($annotation->name);
                    }

                    $rpcParameter->addComment($annotation->name, $annotation->comment);
                    $rpcParameter->addDeclareName($annotation->name);
                } elseif ($annotation instanceof RpcHeader) {
                    $pathItem->addHeader($annotation->name);
                } elseif ($annotation instanceof RpcFormat) {
                    switch ($annotation->format) {
                        case "Json":
                            // default is json
                            break;
                        case "Raw":
                            $defaultResponse->setBody(Body::newRawBody(""));
                            break;
                        default:
                            throw new RuntimeException("format not support: ".$annotation->format);
                    }

                    $pathItem->setFormat($annotation->format);
                } elseif ($annotation instanceof RpcResponseHeader) {
                    $pathItem->addResponseHeader($annotation->name, $annotation->value);
                } elseif ($annotation instanceof RpcAuthority) {
                    $pathItem->addAuthority($annotation->need);
                } elseif ($annotation instanceof RpcMessageHook) {
                    // 兼容旧websocket申明
                    if (strpos($annotation->name, "ws.") === 0) {
                        $websocket->setClass($className);
                        $websocket->setMethod($methodName);
                        $websocket->setNeedConnectMessage(true);
                        $websocket->setNeedDisconnectMessage(true);
                        $websocket->setRoomName(substr($annotation->name, 3));
                    } else {
                        $messageItem->setClass($className);
                        $messageItem->setMethod($methodName);
                        $messageItem->setMessageHookName($annotation->name);
                    }
                } elseif ($annotation instanceof RpcWebsocket) {
                    $websocket->setClass($className);
                    $websocket->setMethod($methodName);
                    $websocket->setNeedConnectMessage($annotation->needConnectMessage);
                    $websocket->setNeedDisconnectMessage($annotation->needDisconnectMessage);
                    $websocket->setRoomName($annotation->roomName);
                } elseif ($annotation instanceof RpcDisableTypeValidation) {
                    if (!$annotation->check_validation) {
                        $rpcParameter->disableValidationCheck();
                    }

                    if (!$annotation->check_type) {
                        $rpcParameter->disableTypeCheck();
                    }
                } elseif ($annotation instanceof RpcCron) {
                    $cronItem->setClass($className);
                    $cronItem->setMethod($methodName);
                    $cronItem->setSpec($annotation->spec, $annotation->every, $annotation->entry);
                    $cronItem->setCronHookName($classMethodMapKey);
                    $this->config->addCron($cronItem);
                }
            }

            $pathItem->addResponseHeader('X-Process', 'backend/'.config('rpc-server.service_name')."/1");
            if ($pathItem->getHttpMethod()) {
                if (!$pathItem->getFormat()) {
                    $pathItem->setFormat("Json");
                }
                $this->config->addPath($pathItem);
            }

            if (!$messageItem->isEmpty()) {
                $this->config->addMessage($messageItem);
            }

            if (!$websocket->isEmpty()) {
                $this->config->addWebsocket($websocket);
            }
        }
    }

    /**
     * @return array
     */
    public function getClassInstancesMap (): array {
        return $this->_classInstancesMap;
    }

    /**
     * @param $className
     * @param $method
     * @return array|null ['methodInstance' => 'ReflectionMethod', parameter => 'RpcParameter']
     */
    public function getMethodItemByClassNameAndMethodName ($className, $method) {
        return $this->_classInstancesMap[$className]['methodItems'][$method] ?? null;
    }
}