<?php


namespace Mainto\MRPC\Server;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Exceptions\RpcMethodNotFoundException;
use Mainto\MRPC\Exceptions\RpcNotSupportException;
use Mainto\MRPC\Protocol\Common\Types\BodyType;
use Mainto\MRPC\Protocol\Request\Extend\RequestExtendUrlInvoke;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Service\RpcParameter;
use ReflectionMethod;

class RpcInvoke {
    public static function invoke (Request $request): Response {
        $classInstancesMap = RpcRouter::getInstance()->getClassInstancesMap();
        if (!$classItem = ($classInstancesMap[$request->getCallClassName()] ?? null)) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $request->getCallClassName(), $request->getCallMethodName()));
        }

        if (!$methodItem = ($classItem['methodItems'][$request->getCallMethodName()] ?? null)) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $request->getCallClassName(), $request->getCallMethodName()));
        }
        $classObj = $classItem['classInstance'];

        /** @var ReflectionMethod $methodInstance */
        /** @var RpcParameter $parameter */
        /** @var Response $response */
        ['methodInstance' => $methodInstance, 'parameter' => $parameter, "defaultResponse" => $response] = $methodItem;

        $params = [];

        if ($request->hasExt()) {
            $ext = $request->getExt();
            if ($ext instanceof RequestExtendUrlInvoke) {
                foreach ($ext->getAllQuery() as $key => $value) {
                    $params[$key] = count($value) === 1 ? $value[0] : $value;
                }
            }
        }

        if ($request->hasBody()) {
            $params += $request->getBody()->toArray();
        }

        $rpcParamsService = new RpcParamsService($parameter, $params);
        $context = new RpcContext($request, $rpcParamsService->getRpcParams());
        $ret = $methodInstance->invokeArgs($classObj, $rpcParamsService->buildMethodParam($context));

        // body
        $body = $response->getBody();
        switch ($body->getBodyType()) {
            case BodyType::JSON:
                if ($ret instanceof Jsonable) {
                    $body->setBodyBytes($ret->toJson());
                } elseif ($ret instanceof Arrayable) {
                    $body->setBodyBytes(json_encode($ret->toArray()));
                } else {
                    $body->setBodyBytes(json_encode($ret));
                }
                break;
            case BodyType::RAW:
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RpcNotSupportException("body type not support: ".$body->getBodyType());
        }

        // responseHeaders
        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        // session
        $context->saveSession();

        return $response;
    }
}