<?php


namespace Mainto\MRPC\Protocol;


use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Request\RequestReader;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendError;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Tool\Net\Conn;
use Throwable;

abstract class BaseRpcConnection {
    protected ?Response $defaultResponse = null;

    protected Conn $conn;

    /**
     * 啰嗦模式
     * @var bool
     */
    public static $verboseMode = false;
    /**
     * @var ConnectionHandler
     */
    private ConnectionHandler $handler;
    private string $streamId;

    protected Response $errResponse;
    protected ResponseExtendError $errResponseExt;

    public function __construct (string $streamId, ConnectionHandler $handler) {
        self::$verboseMode = env('RPC_VERBOSE_MODE', null);

        if (self::$verboseMode === null) {
            self::$verboseMode = env('APP_DEBUG', false);
        }
        $this->handler = $handler;
        $this->streamId = $streamId;

        $this->errResponse = new Response();
        $this->errResponse->useReturnErrType();
        $this->errResponseExt = new ResponseExtendError();
        $this->errResponse->setExtend($this->errResponseExt);
    }

    private function handleRequest (Request $request): Response {
        try {
            return $this->handler->onRequest($request);
        } catch (Throwable $exception) {
            $this->errResponseExt->setTraceId(isset($request) ? $request->getTraceId() : "");
            $this->errResponseExt->setErrMsg($exception->getMessage());

            if ($exception instanceof ValidationException) {
                $this->errResponseExt->setErrorCode(422);
            } elseif ($exception instanceof InvalidArgumentException) {
                $this->errResponseExt->setErrorCode(500);
            } else {
                $this->errResponseExt->setErrorCode(intval($exception->getCode()) ?: 500);
            }

            Log::error($exception, ["streamId" => $this->streamId]);
            return $this->errResponse;
        }
    }

    public function start () {
        while (true) {
            try {
                $request = RequestReader::ReadRequestFrom($this->conn)->toRequest();
            } catch (Throwable $exception) {
                Log::error($exception, ["streamId" => $this->streamId]);
                break;
            }

            $this->dumpRequest($request);

            $startTime = microtime(true);

            $response = $this->handleRequest($request);

            $response->setRuntime(intval((microtime(true) - $startTime) * 1000000));

            $this->dumpResponse($response);

            $response->writeTo($this->conn);
            $response->resetBodyAndExt();
        }

        $this->close();
    }

    public function close () {
        $this->handler->onClose($this->conn);

        $this->conn->close();
    }

    protected function dumpRequest (Request $request) {
        Log::debug("[{$this->streamId}] \n".$request->dump());
    }

    protected function dumpResponse (Response $response) {
        Log::debug("[{$this->streamId}] \n".$response->dump());
    }

    protected function serviceDefaultResponse () {
        if ($this->defaultResponse === null) {
            $this->defaultResponse = new Response();
            $this->defaultResponse->useReturnOKType();
        }

        return $this->defaultResponse;
    }

    /**
     * RpcServer destruct
     */
    public function __destruct () {
        Log::error("server exit...");
    }
}