<?php


namespace Mainto\MRPC\Server;


use BadMethodCallException;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Protocol\Common\Types\BodyType;
use Mainto\MRPC\Protocol\Request\Extend\RequestExtendUrlInvoke;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Service\RpcParameter;
use Mainto\MRPC\Tool\Bytes\Buffer;
use ReflectionMethod;
use RuntimeException;

class RpcInvoke {
    private static ?Buffer $bodyBuffer = null;

    public static function getBodyBuffer (): Buffer {
        if (self::$bodyBuffer === null) {
            self::$bodyBuffer = new Buffer();
        }

        return self::$bodyBuffer;
    }

    public static function invoke (Request $request): Response {
        $classInstancesMap = RpcRouter::getInstance()->getClassInstancesMap();
        if (!$classItem = ($classInstancesMap[$request->getCallClassName()] ?? null)) {
            throw new BadMethodCallException(sprintf("method not found %s::%s", $request->getCallClassName(), $request->getCallMethodName()));
        }

        if (!$methodItem = ($classItem['methodItems'][$request->getCallMethodName()] ?? null)) {
            throw new BadMethodCallException(sprintf("method not found %s::%s", $request->getCallClassName(), $request->getCallMethodName()));
        }
        $classObj = $classItem['classInstance'];

        /** @var ReflectionMethod $methodInstance */
        /** @var RpcParameter $parameter */
        /** @var Response $response */
        ['methodInstance' => $methodInstance, 'parameter' => $parameter, "defaultResponse" => $response] = $methodItem;

        $params = collect([]);

        if ($request->hasExt()) {
            $ext = $request->getExt();
            if ($ext instanceof RequestExtendUrlInvoke) {
                foreach ($ext->getAllQuery() as $key => $value) {
                    $params->offsetSet($key, count($value) === 1 ? $value[0] : $value);
                }
            }
        }

        if ($request->hasBody()) {
            $params = $params->merge($request->getBody()->toCollection());
        }

        $rpcParamsService = new RpcParamsService($parameter, $params);
        $context = new RpcContext($request, $rpcParamsService->getRpcParams());
        $ret = $methodInstance->invokeArgs($classObj, $rpcParamsService->buildMethodParam($context));
        $body = $response->getBody();
        switch ($body->getBodyType()) {
            case BodyType::JSON:
                if ($ret instanceof Jsonable) {
                    $body->setBodyBytes($ret->toJson());
                } elseif ($ret instanceof Arrayable) {
                    $body->setBodyBytes(json_encode($ret->toArray()));
                } else {
                    $body->setBodyBytes(json_encode($ret));
                }
                break;
            case BodyType::RAW:
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RuntimeException("body type not support: ".$body->getBodyType());
        }

        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        return $response;
    }
}