<?php


namespace Mainto\MRPC\Server;


use BadMethodCallException;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Protocol\Common\Types\BodyType;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Service\RpcParameter;
use Mainto\MRPC\Tool\Bytes\Buffer;
use ReflectionMethod;
use RuntimeException;

class RpcInvoke {
    private static ?Buffer $bodyBuffer = null;

    public static function getBodyBuffer (): Buffer {
        if (self::$bodyBuffer === null) {
            self::$bodyBuffer = new Buffer();
        }

        return self::$bodyBuffer;
    }

    public static function invoke (Request $request): Response {
        $classInstancesMap = RpcRouter::getInstance()->getClassInstancesMap();
        if (!$classItem = ($classInstancesMap[$request->getCallClassName()] ?? null)) {
            throw new BadMethodCallException();
        }

        if (!$methodItem = ($classItem['methodItems'][$request->getCallMethodName()] ?? null)) {
            throw new BadMethodCallException();
        }
        $classObj = $classItem['classInstance'];

        /** @var ReflectionMethod $methodInstance */
        /** @var RpcParameter $parameter */
        /** @var Response $response */
        ['methodInstance' => $methodInstance, 'parameter' => $parameter, "defaultResponse" => $response] = $methodItem;

        $rpcParams = new RpcParams($parameter, $request->hasBody() ? $request->getBody()->toCollection() : collect([]));
        $context = new RpcContext($request, $rpcParams);

        $ret = $methodInstance->invokeArgs($classObj, $rpcParams->buildMethodParam($context));
        $body = $response->getBody();
        switch ($body->getBodyType()) {
            case BodyType::JSON:
                if ($ret instanceof Jsonable) {
                    $body->setBodyBytes($ret->toJson());
                } elseif ($ret instanceof Arrayable) {
                    $body->setBodyBytes(json_encode($ret->toArray()));
                } else {
                    $body->setBodyBytes(json_encode($ret));
                }
                break;
            case BodyType::RAW:
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RuntimeException("body type not support: ".$body->getBodyType());
        }


        return $response;
    }
}