<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\RpcServer\Exceptions\RpcMethodNotFoundException;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\Protocol\Common\Types\BodyType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendUrlInvoke;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\Service\RpcParameter;
use ReflectionMethod;

class RpcInvoke {
    public static function invoke (Request $request): Response {
        $classInstancesMap = RpcRouter::getInstance()->getClassInstancesMap();
        $fullClassName = $request->getCallClassName().'Controller';
        if (!$classItem = ($classInstancesMap[$fullClassName] ?? null)) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $fullClassName, $request->getCallMethodName()));
        }

        if (!$methodItem = ($classItem['methodItems'][$request->getCallMethodName()] ?? null)) {
            throw new RpcMethodNotFoundException(sprintf("%s::%s", $fullClassName, $request->getCallMethodName()));
        }
        $classObj = $classItem['classInstance'];

        /** @var ReflectionMethod $methodInstance */
        /** @var RpcParameter $parameter */
        /** @var Response $response */
        ['methodInstance' => $methodInstance, 'parameter' => $parameter, "defaultResponse" => $response] = $methodItem;

        $params = [];

        if ($request->hasExt()) {
            $ext = $request->getExt();
            if ($ext instanceof RequestExtendUrlInvoke) {
                foreach ($ext->getAllQuery() as $key => $value) {
                    $params[$key] = count($value) === 1 ? $value[0] : $value;
                }
            }
        }

        if ($request->hasBody()) {
            $params += (array)$request->getBody()->getContent();
        }

        $rpcParamsService = new RpcParamsService($parameter, $params);
        $context = new RpcContext($request, $rpcParamsService->getRpcParams());
        $ret = $methodInstance->invokeArgs($classObj, $rpcParamsService->buildMethodParam($context));

        // session
        $context->saveSession();

        if ($ret instanceof Response) {
            return $ret;
        }

        // body
        $body = $response->getBody();
        switch ($body->getBodyType()) {
            case BodyType::JSON:
                if ($ret instanceof Jsonable) {
                    $body->setBodyBytes($ret->toJson());
                } elseif ($ret instanceof Arrayable) {
                    $body->setBodyBytes(json_encode($ret->toArray()));
                } else {
                    $body->setBodyBytes(json_encode($ret));
                }
                break;
            case BodyType::RAW:
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RpcNotSupportException("body type not support: ".$body->getBodyType());
        }

        // responseHeaders
        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        return $response;
    }
}