<?php


namespace Mainto\MRPC\Protocol;


use Illuminate\Support\Facades\Log;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Request\Request;
use Mainto\MRPC\Protocol\Request\RequestReader;
use Mainto\MRPC\Protocol\Response\Response;
use Mainto\MRPC\Tool\Net\Conn;
use Throwable;

abstract class BaseRpcConnection {
    protected ?Response $defaultResponse = null;

    protected Conn $conn;

    /**
     * 啰嗦模式
     * @var bool
     */
    public static $verboseMode = false;
    /**
     * @var ConnectionHandler
     */
    private ConnectionHandler $handler;
    private string $streamId;

    public function __construct (string $streamId, ConnectionHandler $handler) {
        self::$verboseMode = env('RPC_VERBOSE_MODE', null);

        if (self::$verboseMode === null) {
            self::$verboseMode = env('APP_DEBUG', false);
        }
        $this->handler = $handler;
        $this->streamId = $streamId;
    }

    public function start () {
        while (true) {
            try {
                $request = RequestReader::ReadRequestFrom($this->conn)->toRequest();
                $this->dumpRequest($request);

                $response = $this->handler->onRequest($request);

                $this->dumpResponse($response);
                $this->conn->write($response->encodeAndResetBody());
            } catch (Throwable $exception) {
                Log::error($exception, ["streamId" => $this->streamId]);

                $response = new Response();
                $response->useReturnErrType();
                $response->setBody(Body::newJsonBody(sprintf('{"msg": %s}', $exception->getMessage())));

                $this->conn->write($response->encodeAndResetBody());
                break;
            }
        }

        $this->close();
    }

    public function close () {
        $this->handler->onClose($this->conn);

        $this->conn->close();
    }

    protected function dumpRequest (Request $request) {
        Log::debug("[{$this->streamId}] req: ".$request->toJson(JSON_UNESCAPED_UNICODE));
    }

    protected function dumpResponse (Response $response) {
        Log::debug("[{$this->streamId}] res: ".$response->toJson(JSON_UNESCAPED_UNICODE));
    }

    protected function serviceDefaultResponse () {
        if ($this->defaultResponse === null) {
            $this->defaultResponse = new Response();
            $this->defaultResponse->useReturnOKType();
        }

        return $this->defaultResponse;
    }

    /**
     * RpcServer destruct
     */
    public function __destruct () {
        Log::error("server exit...");
    }
}