<?php


namespace Mainto\MRPC\Server;


use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Mainto\MRPC\Exceptions\RpcRuntimeException;
use Mainto\MRPC\Service\RpcParameter;

class RpcParams extends Collection{
    /**
     * @var RpcParameter
     */
    private RpcParameter $parameter;

    public function __construct (RpcParameter $parameter, Collection $params) {
        $this->parameter = $parameter;

        parent::__construct($params);

        $this->paramsTypeValidate();
        $this->paramsDataValidate();
    }

    /**
     * 获得声明中的参数
     */
    public function onlyDeclare () {
        return $this->only($this->parameter->getDeclareName());
    }

    /**
     * 构造方法参数列表
     * @param RpcContext $context
     * @return array
     */
    public function buildMethodParam (RpcContext $context) {
        $methodParams = [];
        foreach ($this->parameter->getMethodParameters() as $key => $parameter) {

            if ($this->parameter->getContextPos() === $key) {
                // 兼容模式
                if ($parameter->getType()->getName() !== RpcContext::class) {
                    $ctxClass = $parameter->getType()->getName();
                    $methodParams[] = new $ctxClass($context->getRequest(), $context->getRpcParams());
                } else {
                    $methodParams[] = $context;
                }
            } else {
                $name = $parameter->getName();

                if ($this->has($name)) {
                    $methodParams[] = $this->get($name);
                } elseif(isset($this->parameter->getDefaultMap()[$name])) {
                    $methodParams[] = $this->parameter->getDefaultMap()[$name];
                }
            }
        }

        return $methodParams;
    }

    /**
     * 检测类型
     *
     * @param $name
     * @param $type
     * @return bool
     */
    private function checkType ($name, $type): bool {
        $value = $this->get($name);

        // 检查可空的类型
        if ($value === null && ($this->parameter->getNullableMap()[$name] ?? false)) {
            return true;
        }

        switch ($type) {
            case "string":
                if (is_string($value)) {
                    $this->offsetSet($name, strval($value));
                    return true;
                }
                return false;
            case "uint":
                if (is_numeric($value) && $value >= 0) {
                    $this->offsetSet($name, intval($value));
                    return true;
                }
                return false;
            case "int":
                if (is_numeric($value)) {
                    $this->offsetSet($name, intval($value));
                    return true;
                }
                return false;
            case "double":
                if (is_numeric($value)) {
                    $this->offsetSet($name, floatval($value));
                    return true;
                }
                return false;
            case "unsignedDouble":
                if (is_numeric($value) && $value >= 0) {
                    $this->offsetSet($name, floatval($value));
                    return true;
                }
                return false;
            case "bool":
                if (is_bool($value)) {
                    return true;
                } else {
                    if (is_string($value) || is_numeric($value)) {
                        switch (strtolower($value)) {
                            case 'true':
                            case 1:
                                $this->offsetSet($name, true);
                                return true;
                            case 'false':
                            case 0:
                                $this->offsetSet($name, false);
                                return true;
                        }
                    }
                    return false;
                }
            case "array":
                return is_array($value);
            case "array<int>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (int)$val;
                    }
                    $this->offsetSet($name, $value);
                    return true;
                }
                return false;
            case  "array<uint>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $val = (int)$val;
                        if ($val < 0) return false;

                        $value[$key] = (int)$val;
                    }
                    $this->offsetSet($name, $value);
                    return true;
                }
                return false;
            case  "array<string>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (string)$val;
                    }
                    $this->offsetSet($name, $value);
                    return true;
                }
                return false;
            case "mixed":
                return true;
        }

        return false;
    }

    private function paramsTypeValidate () {
        if ($this->parameter->isEnableTypeCheck()) {
            foreach ($this->parameter->getTypeMap() as $key => $type) {
                if ($this->has($key)) {
                    if (!$this->checkType($key, $type)) {
                        throw new RpcRuntimeException("key `{$key}` is not a {$type} instance", 422);
                    }
                }
            }
        }
    }

    /**
     * @throws ValidationException
     */
    private function paramsDataValidate () {
        if ($this->parameter->isEnableValidationCheck() && ($validationMap = $this->parameter->getValidationMap())) {
            foreach ($this->parameter->getMustHave() as $item) {
                if (!isset($validationMap[$item])) {
                    $validationMap[$item] = "required";
                }
            }
            $validator = Validator::make($this->all(), $validationMap, [], []);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
        }
    }
}