<?php


namespace Mainto\MRPC\Protocol\Request;


use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Exceptions\RpcRuntimeException;
use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\IO\IO;
use Mainto\MRPC\Tool\IO\Writer;
use Mainto\MRPC\Tool\IO\WriterTo;

class Request implements WriterTo, Jsonable {
    private string $traceId = "";
    private string $callClassName = "";
    private string $callMethodName = "";
    private int $type = 0;
    private ?Extend $ext = null;
    private ?Body $body = null;
    private bool $isStream = false;

    /**
     * @param string $traceId
     */
    public function setTraceId (string $traceId): void {
        $this->traceId = $traceId;
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->traceId;
    }

    private function encodeHeaderLen () {
        if (strlen($this->callClassName) > 255 || strlen($this->callMethodName) > 255) {
            throw new RpcRuntimeException("header too large.");
        }

        $size = 0;
        $size += 2;  // head size
        $size += 2;  // version
        $size += 2;  // extend magic number
        $size += 1;  // has body
        $size += 8;  // 预留
        $size += 2;  // stream | empty  | main type | second type
        $size += 16; // request trace id

        $size += 1; // class size
        $size += strlen($this->callClassName);
        $size += 1; // method size
        $size += strlen($this->callMethodName);

        // uint16
        return $size;
    }

    private function encodeHeader () {
        $size = $this->encodeHeaderLen();
        $buf = Binary::makeBytes($size);
        Binary::$littleEndian::putUint16($buf, 0, $size);

        $buf[2] = Version::ProtocolVersion[0];
        $buf[3] = Version::ProtocolVersion[1];
        if ($this->ext !== null) {
            Binary::$littleEndian::putUint16($buf, 4, $this->ext->magicNum());
        }

        if ($this->body !== null) {
            $buf[6] = "\x01";
        }

        // type

        if ($this->isStream) {
            $this->type = $this->type | BaseType::StreamOpen | $this->type;
        }

        Binary::$littleEndian::putUint16($buf, 15, $this->type);

        Binary::copy($buf, 17, 33, $this->traceId);

        $classLen = strlen($this->callClassName);
        Binary::putUint8($buf, 33, $classLen);
        Binary::copy($buf, 34, 34 + $classLen, $this->callClassName);

        $methodLen = strlen($this->callMethodName);
        Binary::putUint8($buf, 34 + $classLen, $methodLen);
        Binary::copy($buf, 34 + $classLen + 1, 34 + $classLen + 1 + $methodLen, $this->callMethodName);

        return $buf;
    }

    public function encode() {
        return IO::IOWriterToBytes($this);
    }

    public function writeTo (Writer $w): int {
        $b = $this->encodeHeader();
        $totalWrite = $w->write($b);

	    if ($this->ext !== null && $this->ext instanceof HeaderExtend) {
	        $totalWrite += $this->ext->writeHeaderTo($w);
        }

        if ($this->body !== null) {
            $totalWrite += $this->body->writeTo($w);
        }

        if ($this->ext !== null && $this->ext instanceof BodyExtend) {
            $totalWrite += $this->ext->writeBodyTo($w);
        }


        return $totalWrite;
    }

    /**
     * @param string $class
     * @param string $method
     */
    public function setCallClassAndMethod (string $class, string $method): void {
        $this->callClassName = $class;
        $this->callMethodName = $method;
    }

    public function setTypeWithExtend (int $getType, ?Extend $extent) {
        $this->type = $getType;
        $this->ext = $extent;
    }

    public function setExtend (?Extend $extent) {
        $this->ext = $extent;
    }

    public function setBody (?Body $body) {
        $this->body = $body;
    }

    /**
     * @return Body|null
     */
    public function getBody (): ?Body {
        return $this->body;
    }

    public function checkParams() {
        if (strlen($this->callClassName) > 255 || strlen($this->callMethodName) > 255) {
            throw new RpcRuntimeException("header too large.");
        }

        if ($this->body !== null) {
            $this->body->checkParams();
        }

        if ($this->ext !== null) {
            $this->ext->checkParams();
        }
    }

    public function setType (int $type) {
        $this->type = $type;
    }

    /**
     * @return string
     */
    public function getType (): string {
        return $this->type;
    }

    public function toJson ($options = 0) {
        return json_encode([
            "type"           => BaseType::Map[$this->type],
            "traceId"        => $this->getTraceId(),
            "callClassName"  => $this->getCallClassName(),
            "callMethodName" => $this->getCallMethodName(),
            "body"           => optional($this->getBody())->toJson($options),
            //            "ext"           => $this->getExtend()->toJson($options),
        ], $options);
    }

    /**
     * @return string
     */
    public function getCallClassName (): string {
        return $this->callClassName;
    }

    /**
     * @return string
     */
    public function getCallMethodName (): string {
        return $this->callMethodName;
    }
}