<?php


namespace Mainto\MRPC\Protocol\Response;


use Illuminate\Contracts\Support\Jsonable;
use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Types\ResponseHeaderType;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\IO\IO;
use Mainto\MRPC\Tool\IO\Writer;
use Mainto\MRPC\Tool\IO\WriterTo;

class Response implements WriterTo, Jsonable {
    private string $type;
    private ?Extend $ext = null;
    private ?Body $body = null;
    private bool $isStream = false;

    public function setTypeWithExtend (int $getType, ?Extend $extent) {
        $this->type = $getType;
        $this->ext = $extent;
    }

    public function setExtend (?Extend $extent) {
        $this->ext = $extent;
    }

    public function setBody (?Body $body) {
        $this->body = $body;
    }

    public function useReturnOKType () {
        $this->type = ResponseHeaderType::ReturnOKType;
        return $this;
    }

    public function useReturnAuthInvokeType () {
        $this->type = ResponseHeaderType::ReturnAuthInvokeType;
        return $this;
    }

    public function useReturnAuthRevokeType () {
        $this->type = ResponseHeaderType::ReturnAuthRevokeType;
        return $this;
    }

    public function useReturnErrType () {
        $this->type = ResponseHeaderType::ReturnErrType;
        return $this;
    }

    public function useReturnSystemErrType () {
        $this->type = ResponseHeaderType::ReturnSystemErrType;
        return $this;
    }

    private function encodeHeaderLen () {
        $size = 0;
        $size += 2;  // head size
        $size += 2;  // version
        $size += 2;  // extend magic number
        $size += 1;  // has body
        $size += 8;  // 预留
        $size += 2;  // stream | empty  | main type | second type

        // uint16
        return $size;
    }

    private function encodeHeader () {
        $size = $this->encodeHeaderLen();
        $buf = Binary::makeBytes($size);
        Binary::$littleEndian::putUint16($buf, 0, $size);

        $buf[2] = Version::ProtocolVersion[0];
        $buf[3] = Version::ProtocolVersion[1];
        if ($this->ext !== null) {
            Binary::$littleEndian::putUint16($buf, 4, $this->ext->magicNum());
        }

        if ($this->body !== null) {
            $buf[6] = "\x01";
        }

        // type
        if ($this->isStream) {
            $this->type = $this->type | BaseType::StreamOpen | $this->type;
        }

        Binary::$littleEndian::putUint16($buf, 15, $this->type);

        return $buf;
    }

    public function encode() {
        return IO::IOWriterToBytes($this);
    }

    public function encodeAndResetBody() {
        $bytes = IO::IOWriterToBytes($this);

        optional($this->getBody())->reset();

        return $bytes;
    }

    public function writeTo (Writer $w): int {
        $b = $this->encodeHeader();
        $totalWrite = $w->write($b);

        if ($this->ext !== null) {
            if ($this->ext instanceof HeaderExtend) {
                $totalWrite += $this->ext->writeHeaderTo($w);
            }
        }

        if ($this->body !== null) {
            $totalWrite += $this->body->writeTo($w);
        }

        if ($this->ext !== null) {
            if ($this->ext instanceof BodyExtend) {
                $totalWrite += $this->ext->writeBodyTo($w);
            }
        }

        return $totalWrite;
    }

    public function getBody (): ?Body {
        return $this->body;
    }

    public function toJson ($options = 0) {
        return json_encode([
           "type" => BaseType::Map[$this->type],
           "body" => optional($this->getBody())->toJson($options)
        ], $options);
    }
}