<?php


namespace Mainto\MRPC\Protocol\Response;


use Mainto\MRPC\Protocol\Common\BaseType;
use Mainto\MRPC\Protocol\Common\Body;
use Mainto\MRPC\Protocol\Common\Version;
use Mainto\MRPC\Tool\Binary;
use Mainto\MRPC\Tool\Bytes\Bytes;
use Mainto\MRPC\Tool\IO\IOUtil;
use Mainto\MRPC\Tool\IO\Reader;
use RuntimeException;

class ResponseReader {
    private Reader $reader;

    private bool $hasBody;
    private bool $readBody;

    private int $magicNum;

    private int $type;
    private bool $isStream;

    private function __construct (Reader $reader) {
        $this->reader = $reader;
    }

    public static function ReadResponseFrom (Reader $reader) {
        $r = new self($reader);

        $r->init();

        return $r;
    }

    private function init () {
        $_ = $this->readerHeader();

        $this->readBody = !$this->hasBody;
    }

    private function readerHeader () {
        $headerSize = $this->readFull(2);

        $size = Binary::$littleEndian::strToUint16($headerSize) - 2;

        if ($size > 542) { // max size 544 - 2
            throw new RuntimeException("ErrHeaderTooLarge");
        }

        $headerByte = $this->readFull($size);

        if (!Bytes::equal(substr($headerByte, 0, 2), Version::ProtocolVersion)) {
            throw new RuntimeException("ErrProtocolVersion");
        }

        $this->magicNum = Binary::$littleEndian::strToUint16(substr($headerByte, 2, 2));
        $this->hasBody = $headerByte[4] === "\x01";

        $this->type = Binary::$littleEndian::strToUint16(substr($headerByte, 13, 2)) & BaseType::TypeMask;

        $this->isStream = ($this->type & BaseType::TypeMask) !== "\x00";

        return $size + 2;
    }

    private function readFull ($size) {
        return IOUtil::readFullSize($this->reader, $size);
    }

    public function getBody () {
        if (!$this->hasBody) {
            return null;
        }

        $this->readBody = true;

        return Body::newBodyFromReader($this->reader);
    }

    public function getExtend (): ?Extend {
        if ($this->magicNum === 0) {
            return null;
        }
        if (!$this->readBody) {
            throw new RuntimeException("ErrMustReadBodyFirst");
        }

        $extend = ExtendFactory::getManagedRequestExtendByMagicNum($this->magicNum);
        $extend->ReadFrom($this->reader);

        return $extend;
    }

    /**
     * @return int
     */
    public function getType (): int {
        return $this->type;
    }
}