<?php


namespace Mainto\MRPC\Service;


use InvalidArgumentException;
use Mainto\MRPC\Server\RpcContext;
use ReflectionException;
use ReflectionParameter;

class RpcParameter {
    /**
     * 是否开启类型检测
     * @var bool
     */
    private bool $enableTypeCheck = true;

    /**
     * 是否开启验证检测
     * @var bool
     */
    private bool $enableValidationCheck = true;

    /**
     * 所有声明的参数
     * @var array
     */
    private array $declareName = [];

    /**
     * 反射参数内容 方法的参数
     * @var ReflectionParameter[]
     */
    private array $methodParameters = [];

    /**
     * 必须包含的参数
     * @var array
     */
    private array $mustHave = [];

    /**
     * 默认参数映射
     * @var array
     */
    private array $defaultMap = [];

    /**
     * 校验规则映射
     * @var array
     */
    private array $validationMap = [];

    /**
     * 类型Map
     * @var array
     */
    private array $typeMap = [];

    /**
     * 可空的参数Map
     * @var array
     */
    private array $nullableMap = [];

    /**
     * 默认参数数组
     * @var array
     */
    private ?array $defaultArray = null;

    /**
     * 要求注入的context位置
     * @var int|null
     */
    private ?int $contextPos = null;

    /**
     * 注释映射
     * @var array
     */
    private array $commentMap = [];

    /**
     * 增加一个参数
     * @param ReflectionParameter $parameter
     * @throws ReflectionException
     */
    public function addParameter (ReflectionParameter $parameter) {
        $name = $parameter->getName();
        $this->methodParameters[] = $parameter;

        if (!$parameter->isDefaultValueAvailable()) {
            if ($parameter->getType() && $parameter->getType()->getName() == RpcContext::class) {
                if ($this->contextPos !== null) {
                    throw new InvalidArgumentException("request only have once");
                }

                $this->contextPos = count($this->methodParameters) - 1;
            } else {
                $this->mustHave[] = $name;
            }
        } else {
            $this->defaultMap[$name] = $parameter->getDefaultValue();
        }
    }

    /**
     * 当前参数列表是否为空
     * @return bool
     */
    public function isEmpty () {
        return count($this->methodParameters) == 0;
    }

    /**
     * 设置某个参数为必须
     * @param string $paramName
     */
    public function setRequired (string $paramName) {
        if (!in_array($paramName, $this->mustHave)) {
            $this->mustHave[] = $paramName;
        }
    }

    /**
     * 设置校验器
     * @param $name
     * @param $validation
     */
    public function setValidation ($name, $validation) {
        $this->validationMap[$name] = $validation;
    }

    /**
     * 设置类型
     * @param $name
     * @param $type
     */
    public function setType ($name, $type) {
        $this->typeMap[$name] = $type;
    }

    /**
     * 设置可空
     * @param $name
     * @param $type
     */
    public function setNullable ($name) {
        $this->nullableMap[$name] = true;
    }

    /**
     * 设置某个参数的默认值
     * @param string $paramName
     * @param $default
     */
    public function setDefault (string $paramName, $default) {
        $this->defaultMap[$paramName] = $default;

        if (($index = array_search($paramName, $this->mustHave)) !== false) {
            unset($this->mustHave[$index]);
        }
    }

    /**
     * 关闭类型检测
     */
    public function disableTypeCheck () {
        $this->enableTypeCheck = false;
    }

    /**
     * 关闭类型检测
     */
    public function disableValidationCheck () {
        $this->enableValidationCheck = false;
    }

    /**
     * 添加一个声明的对象名称
     * @param $name
     */
    public function addDeclareName ($name) {
        $this->declareName[] = $name;
    }

    /**
     * 添加一个声明的对象名称
     * @param $name
     * @param $value
     */
    public function addComment ($name, $value) {
        $this->commentMap[$name] = $value;
    }

    /**
     * 获得所有声明的名称
     * @return array
     */
    public function getDeclareName (): array {
        return $this->declareName;
    }

    /**
     * @return bool
     */
    public function isEnableTypeCheck (): bool {
        return $this->enableTypeCheck;
    }

    /**
     * @return array
     */
    public function getTypeMap (): array {
        return $this->typeMap;
    }

    /**
     * @return ReflectionParameter[]
     */
    public function getMethodParameters (): array {
        return $this->methodParameters;
    }

    /**
     * @return int|null
     */
    public function getContextPos (): ?int {
        return $this->contextPos;
    }

    /**
     * @return array
     */
    public function getDefaultMap (): array {
        return $this->defaultMap;
    }

    /**
     * @return bool
     */
    public function isEnableValidationCheck (): bool {
        return $this->enableValidationCheck;
    }

    /**
     * @return array
     */
    public function getValidationMap (): array {
        return $this->validationMap;
    }

    /**
     * @return array
     */
    public function getDefaultArray (): array {
        return $this->defaultArray;
    }

    /**
     * @return array
     */
    public function getMustHave (): array {
        return $this->mustHave;
    }

    /**
     * @return array
     */
    public function getNullableMap (): array {
        return $this->nullableMap;
    }
}