<?php

namespace Mainto\RpcTool\Command;

use Illuminate\Console\Command;
use Illuminate\Support\Str;

class RpcSdkJavaDump extends Command {
    protected $signature = 'rpc:sdk-java {--dump_path=}';

    private $definition;

    private string $basePackageName;

    public function handle () {
        $this->definition = json_decode(file_get_contents(__DIR__.'/source.json'), true);

        $dumpPath = value_if($this->option('dump_path'), format_path('.'));

        $serviceName = $this->definition['serviceName'];

        $this->basePackageName = "com.mainto.bridge.".Str::snake($serviceName);

        $basePath = $dumpPath."/micro-bridge/".Str::snake($serviceName, "-")."/src/main/java/".str_replace(".", '/', $this->basePackageName);

        $structPath = $basePath."/struct";
        foreach ($this->definition['structs'] as $struct) {
            if (Str::endsWith($struct['name'], "Params") && Str::contains($struct['name'], "@")) {
                continue;
            }

            $newStructName = $this->newStructName($struct);
            $shortName = str_replace($this->basePackageName.".struct.", "", $newStructName);

            $fileName = $structPath."/".str_replace(".", "/", $shortName).".java";

            if (!file_exists(dirname($fileName))) {
                mkdir(dirname($fileName), 0777, true);
            }
            file_put_contents($fileName, view("java/java_struct", [
                'className'  => substr($struct['name'], strrpos($struct['name'], ".") + 1),
                'properties' => array_map(function ($item) {
                    return [
                        'name'                  => $item['name'],
                        'type'                  => $this->typeConvert($item['type']),
                        'defaultValueAvailable' => $item['defaultValueAvailable'],
                        'default'               => $item['default'] ?? "",
                    ];
                }, $struct['properties']),
                'package'    => substr($newStructName, 0, strrpos($newStructName, ".")),
            ])->render());
        }

        $invokePath = $basePath."/invoker";
        foreach ($this->definition['controllers'] as $controller) {
            $fileName = $invokePath."/".str_replace(".", '/', $controller['registerName']).".java";
            if (!file_exists(dirname($fileName))) {
                mkdir(dirname($fileName), 0777, true);
            }

            file_put_contents($fileName, view("java/java_invoker", [
                'serviceNameSnake' => Str::snake($serviceName),
                'serviceName'      => $serviceName,
                'registerName'     => $controller['registerName'],
                'className'        => $controller['name'],
                "namespace"        => implode('.', array_map(function ($item) {
                    return Str::snake($item);
                }, explode('.', substr($controller['registerName'], 0, strrpos($controller['registerName'], "."))))),
                "methods"          => array_map(function ($method) {

                    $responseType = $this->typeConvert($method["responseType"]);
                    if (Str::startsWith($responseType, "Map<")) {
                        $decodeResponseType = "new TypeToken<$responseType>() {}";
                    } else {
                        $decodeResponseType = $responseType.".class";
                    }

                    return [
                        "name"               => $method["name"],
                        "responseType"       => $responseType,
                        "decodeResponseType" => $decodeResponseType,
                        "parameterStr"       => $this->parseParameter($method["requestType"]),
                        "params"             => $this->parseParams($method['requestType']),
                    ];
                }, $controller['methods']),

            ])->render());
        }

        $gradleFile = $dumpPath."/micro-bridge/".Str::snake($serviceName, "-")."/build.gradle";

        file_put_contents($gradleFile, view("java/build_gradle", [
            "group"   => "com.mainto.bridge",
            "version" => $this->definition["version"],
        ])->render());

        $settingsFile = $dumpPath."/micro-bridge/".Str::snake($serviceName, "-")."/settings.gradle";
        file_put_contents($settingsFile, "rootProject.name = '".Str::snake($serviceName, "-")."'");
    }

    public function parseParams (string $requestType) {
        if ($requestType == "EmptyParams") {
            return "null";
        }

        $struct = $this->getStruct($requestType);
        if (Str::endsWith($requestType, "Params") && Str::contains($requestType, "@")) {
            // parameter mode

            $parameterStr = "Map.of(";

            foreach ($struct["properties"] as $property) {
                $parameterStr .= "\"{$property["name"]}\", {$property["name"]}, ";
            }

            return rtrim($parameterStr, ", ").")";

        } else {
            return Str::camel(substr($struct["name"], strrpos($struct["name"], ".") + 1));
        }
    }

    public function newStructName ($struct) {
        return $this->basePackageName.".struct.".$struct["name"];
    }

    public function parseParameter (string $requestType): string {
        if ($requestType == "EmptyParams") {
            return "";
        }

        $struct = $this->getStruct($requestType);
        if (Str::endsWith($requestType, "Params") && Str::contains($requestType, "@")) {
            // parameter mode

            $parameterStr = "";

            foreach ($struct["properties"] as $property) {
                $parameterStr .= $this->typeConvert($property["type"])." {$property["name"]}, ";
            }

            return rtrim($parameterStr, ", ");

        } else {
            $shortName = Str::camel(substr($struct["name"], strrpos($struct["name"], ".") + 1));
            return $this->newStructName($struct)." $shortName";
        }
    }

    public function typeConvert ($type): string {
        switch ($type) {
            case "bool" :
                return "Boolean";
            case "int" :
            case "uint" :
                return "Integer";
            case "int64" :
                return "Long";
            case "float" :
                return "Float";
            case "string" :
                return "String";
            case "void" :
                return "Void";
            case "mixed" :
                return "Object";
            default:
                if (Str::startsWith($type, "Map<")) {
                    if (preg_match('/Map<(.*)>/', $type, $match) && count($match) > 1) {
                        $types = explode(',', $match[1]);
                        $childType = trim(array_pop($types));
                        $keyType = trim(array_pop($types));
                        return "Map<".$this->typeConvert($keyType).", ".$this->typeConvert($childType).">";
                    }
                }

                if (Str::contains($type, "[]")) {
                    if (preg_match('/(.*)(\[])+/', $type, $match) && count($match) > 1) {
                        return $this->typeConvert($match[1]).$match[2];
                    }
                }

                $struct = $this->getStruct($type);
                if ($struct) {
                    return $this->newStructName($struct);
                }
                return rtrim($type, '!?');
        }
    }

    public function getStruct ($type) {
        $rtrim = rtrim($type, '!?');
        if (isset($this->definition['structs'][$rtrim])) {
            return $this->definition['structs'][$rtrim];
        }

        return null;
    }
}