<?php


namespace Mainto\RpcTool\Document;


use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Support\Str;
use Mainto\RpcServer\Base\RpcEnum;
use Mainto\RpcServer\Exceptions\BaseServiceException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcServer\Definition\Deprecated;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcRouter;
use Mainto\RpcServer\Util\ArrayHelper;
use Mainto\RpcServer\Util\Types\Map;
use Mainto\RpcTool\Document\Definition\Enum;
use Mainto\RpcTool\Document\Definition\EnumUnit;
use Mainto\RpcTool\Document\Definition\Exception;
use Mainto\RpcTool\Document\Definition\ExceptionCode;
use ReflectionClass;
use RuntimeException;

class Document {
    use ArrayHelper;

    /**
     * @var Map<Controller>|Controller[]
     */
    public Map $controllers;
    /**
     * @var Map<Struct>|Struct[]
     */
    public Map $structs;

    /**
     * @var Enum[]
     */
    public array $enums = [];

    /**
     * @var Exception[]
     */
    public array $exceptions = [];

    /**
     * @var string
     */
    public string $serviceName;

    /**
     * @var string
     */
    public string $serviceNameSnake;

    /**
     * @var string
     */
    public string $version;

    /**
     * @var array
     */
    public array $dependencies = [];

    /**
     * @var array
     */
    public array $docs = [];

    /**
     * @var string
     */
    public string $alias = "";

    private array $_exceptionCodes = [];

    public static function fromRpcDefinition (RpcDefinition $definition): Document {
        $doc = new self();
        $definitionRef = new \ReflectionClass($definition);
        foreach ($definitionRef->getProperties(\ReflectionProperty::IS_PUBLIC) as $property) {
            if (property_exists($doc, $property->name)) {
                $setMethod = 'set'.Str::studly($property->name);
                if (method_exists($doc, $setMethod)) {
                    $doc->{$setMethod}($definition->{$property->name});
                } else {
                    $doc->{$property->name} = $definition->{$property->name};
                }
            }
        }

        return $doc;
    }

    public function setControllers (Map $controllers) {
        $this->controllers = new Map();
        foreach ($controllers as $key => $controller) {
            $this->controllers->set($key, Controller::fromDefinitionController($controller));
        }
    }

    /**
     * @param ReflectionClass $reflection
     */
    private function resolveEnum (ReflectionClass $reflection) {
        $enum = new Enum();
        $enum->namespace = $reflection->getNamespaceName();
        $enum->name = $reflection->getName();
        $reflectionClassConstants = $reflection->getReflectionConstants();
        foreach ($reflectionClassConstants as $classConstant) {
            $unit = new EnumUnit();
            $unit->name = $classConstant->name;
            $unit->value = $classConstant->getValue();
            $unit->comment = get_doc($classConstant->getDocComment());
            $enum->enums[] = $unit;
        }
        $annotations = app(AnnotationReader::class)->getClassAnnotations($reflection);
        foreach ($annotations as $annotation) {
            if ($annotation instanceof Alias) {
                $enum->alias = $annotation->name;
            }
        }

        $description = "";
        foreach (explode("\n", $reflection->getDocComment()) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;

                $description .= $_doc."\n";
                continue;
            }
            if (strpos($line, '@deprecated') !== false) {
                $deprecated = new Deprecated();
                $deprecated->reason = trim(substr($line, strpos($line, '@deprecated') + 11));
                $enum->deprecated = $deprecated;
            }

            if (strpos($line, '@since') !== false) {
                $enum->since = trim(substr($line, strpos($line, '@since') + 6));
            }
        }

        $enum->description = rtrim($description, "\n");

        $this->enums[] = $enum;
    }

    /**
     * @param ReflectionClass $reflection
     */
    private function resolveException (ReflectionClass $reflection) {
        $regex = '/@method[ ]*static[ ]*([A-Z0-9a-z_]?[A-Za-z0-9_]*)[ ]*([A-Z0-9_]*)[ ]*\([ ]*\$(code|codeOrText)[ ]*=[ ]*(0x[0-9A-F]{9})[ ]*,[ ]*\$(text)[ ]*=[ ]*[\'"](.*)[\'"][ ]*\)/m';
        if ($reflection->isSubclassOf(BaseServiceException::class)) {
            $comment = '';
            $exceptionClassName = $alias = $reflection->getName();
            $annotations = app(AnnotationReader::class)->getClassAnnotations($reflection);
            foreach ($annotations as $annotation) {
                if ($annotation instanceof Alias) {
                    $alias = $annotation->name;
                }
            }
            $exception = new Exception();
            $exception->alias = $alias;
            $exception->name = $exceptionClassName;

            foreach (explode("\n", $reflection->getDocComment()) as $line) {
                $line = trim($line, " \t\n\r\0\x0B*");

                if (strpos($line, '@method') !== false) {
                    if (preg_match_all($regex, $line, $matches, PREG_SET_ORDER, 0)) {
                        [$_, $exceptionName, $exceptionMethodName, $firstParamName, $exceptionCode, $secondParamName, $message] = $matches[0];
                        $code = intval(substr($exceptionCode, 2), 16);
                        if (isset($this->_exceptionCodes[$code])) {
                            throw new RpcRuntimeException("duplicate exception code on {$reflection->getName()}, code : $exceptionCode");
                        }

                        $this->_exceptionCodes[$code] = true;

                        if ($exceptionName != $reflection->getShortName()) {
                            throw new RuntimeException("doc: {$line} return name is not equals class {$reflection->getName()}", 500);
                        }

                        $exceptionCode = new ExceptionCode();
                        $exceptionCode->code = $code;
                        $exceptionCode->comment = $comment ?: $message;
                        $exceptionCode->message = $message;
                        $exceptionCode->exceptionName = $exceptionMethodName;

                        $exception->codes[] = $exceptionCode;
                        $comment = '';
                    } else {
                        throw new RpcRuntimeException("match failed on on {$reflection->getName()}, line content : $line");
                    }
                }
                if (strpos($line, '@exception-text') !== false) {
                    $comment = trim(str_replace(['@exception-text', '*'], '', $line));
                }
            }

            $this->exceptions[] = $exception;
        }
    }

    public function resolveEnumAndException () {
        $this->enums = [];
        $this->exceptions = [];
        $appDir = app_path();
        if (RpcRouter::getAppDir()) {
            $appDir = RpcRouter::getAppDir();
        }

        $classes = get_classes($appDir);
        $refEnums = $refExceptions = [];
        foreach ($classes as $class) {
            $ref = new ReflectionClass($class);
            if ($ref->isSubclassOf(RpcEnum::class)) {
                $refEnums[] = $ref;
            }

            if ($ref->isSubclassOf(BaseServiceException::class)) {
                $refExceptions[] = $ref;
            }
        }

        foreach ($refEnums as $refEnum) {
            $this->resolveEnum($refEnum);
        }

        foreach ($refExceptions as $refException) {
            $this->resolveException($refException);
        }
    }
}