<?php


namespace Mainto\RpcTool\Command;


use Illuminate\Console\Command;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Parameter;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use Mainto\RpcTool\Document\Document;
use Mainto\RpcTool\Document\Method;

class RpcSdkPHPCommand extends Command {

    protected $signature = 'rpc:sdk-php {option?} {--write_json=} {--dump_path=} {--enum_dump_path=} {--struct_dump_path=} {--controller_dump_path=}';
    private array $objectMap = [];

    private Document $document;

    public function handle () {
        $this->document = Document::fromRpcDefinition(RpcDefinition::getInstance());
        $this->document->resolveEnumAndException();

        switch ($this->argument('option')) {
            case 'controller':
                $this->controller();
                $this->struct();
                break;
            case 'enum':
                $this->enum();
                break;
            default:
                $this->controller();
                $this->struct();
                $this->enum();
        }
        if ($this->option('write_json')) {
            $this->packageJson();
        }
    }

    public function controller () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        foreach ($this->document->controllers as $controller) {
            $registerPath = str_replace('\\', '/', $controller->registerName);
            $controllerDir = $microName;
            if (strrpos($registerPath, "/")) {
                $controllerDir = path_join($controllerDir, dirname($registerPath));
            }

            $namespace = str_replace("/", "\\", $controllerDir);

            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Invokes/",
                $controllerDir,
                basename($registerPath).".php"
            );
            if (!file_exists(dirname($path))) {
                mkdir(dirname($path), 0777, true);
            }

            $fill = [
                'microName'          => $microName,
                'className'          => $controller->registerName,
                'shortClassName'     => class_basename($controller->registerName),
                'methods'            => array_map(function (Method $method) {
                    $requestType = $docRequestType = "";
                    $rpcParams = [];
                    $parametersMap = array_filter($method->getParameters(), function (Parameter $parameter) { return $parameter->type != RpcContext::class; });

                    if ($method->requestType == RpcDefinition::EmptyType) {
                        $isObjectRequest = false;
                    } else {
                        [$requestType, $docRequestType, $isObjectRequest] = $this->parserTypeString($method->requestType);
                        if (isset(Language::$rpcTypeMap[$requestType])) {
                            $requestType = Language::$rpcTypeMap[$requestType];
                        }

                        if (isset(Language::$rpcDocTypeMap[$docRequestType]) && Language::$rpcDocTypeMap[$docRequestType] != $docRequestType) {
                            $docRequestType = Language::$rpcDocTypeMap[$docRequestType];
                        }

                        if (RpcDefinition::getInstance()->inStructCache($method->requestType)) {
                            $struct = RpcDefinition::getInstance()->getStruct($method->requestType);
                            if ($struct instanceof NotDefinitionStruct) {
                                $isObjectRequest = false;
                                $rpcParams = array_map(function (Property $property) use ($method, $parametersMap) {
                                    $isParameter = isset($parametersMap[$property->name]);
                                    $type = $property->type;

                                    if (isset(Language::$rpcTypeMap[$type])) {
                                        $type = Language::$rpcTypeMap[$type];
                                    } else {
                                        $type = 'mixed';
                                    }

                                    return (object)[
                                        'name'        => $property->name,
                                        'comment'     => $property->comment,
                                        'require'     => $property->require,
                                        'type'        => $type,
                                        'isParameter' => $isParameter,
                                    ];
                                }, $struct->properties);
                            }
                        }
                    }

                    [$responseType, $docResponseType, $isObjectResponse] = $this->parserTypeString($method->responseType);
                    if ($isObjectRequest) {
                        /** @var Parameter $objectParameter */
                        $objectParameter = array_first($parametersMap);
                        $parameterStr = ($objectParameter->nullable ? ('?'.$requestType) : $requestType).' $'.$objectParameter->name;
                        if ($objectParameter->defaultValueAvailable) {
                            $parameterStr .= " = ".var_export_min($objectParameter->default);
                        }
                    } else {
                        $parameterStr = "";
                        foreach ($parametersMap as $parameter) {
                            if (isset(Language::$rpcTypeMap[$parameter->type])) {
                                $parameterStr .= Language::$rpcTypeMap[$parameter->type].' $'.$parameter->name;
                            } else {
                                $parameterStr .= '$'.$parameter->name;
                            }

                            $parameterStr .= $parameter->defaultValueAvailable ? ' = '.var_export_min($parameter->default) : '';
                            $parameterStr .= ', ';
                        }

                        $parameterStr = rtrim($parameterStr, ', ');
                    }

                    return (object)[
                        'name'             => $method->name,
                        'description'      => $method->description,
                        'deprecated'       => $method->deprecated,
                        'alias'            => $method->alias,
                        'isObjectRequest'  => $isObjectRequest,
                        'docRequestType'   => $docRequestType,
                        'requestType'      => $requestType,
                        'rpcParams'        => $rpcParams,
                        'isObjectResponse' => $isObjectResponse,
                        'docResponseType'  => $docResponseType,
                        'responseType'     => $responseType,
                        'parameterStr'     => $parameterStr,
                        'parametersMap'    => $parametersMap,
                    ];
                }, $controller->methods->values()),
                'namespace'          => $namespace,
                'mockClassNamespace' => config('rpc-tool.cmd.namespace.mock_class'),
                'rpcClassNamespace'  => config('rpc-tool.cmd.namespace.rpc_class'),
            ];

            file_put_contents($path, "<?php\n".view('bridge-controller', $fill)->render());
        }
    }

    private function parserTypeString ($targetType): array {
        $isObjectType = false;
        $docType = $targetType;
        if (starts_with($targetType, "Map<")) {
            preg_match('/Map<(.*)>/', $targetType, $match);
            $childType = ltrim(explode(',', $match[1])[1]);

            if (RpcDefinition::getInstance()->inStructCache($childType)) {
                $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", '\\'.ltrim($this->getFullStructNameByStruct(
                        RpcDefinition::getInstance()->getStruct($childType)
                    ), '\\'));
                $isObjectType = !(RpcDefinition::getInstance()->getStruct($childType) instanceof NotDefinitionStruct);
            } else {
                $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", $childType);
            }

            $targetType = "\Mainto\RpcServer\Util\Types\Map";
        } else if (ends_with($targetType, "[]")) {
            [$childType, $arrayDeepCount] = ArrayTypeParser::getArrayType($targetType);
            $docType = $childType.str_repeat('[]', $arrayDeepCount);
            if (RpcDefinition::getInstance()->inStructCache($childType)) {
                $childDocType = $this->getFullStructNameByStruct(
                    RpcDefinition::getInstance()->getStruct($childType)
                );
                $docType = $childDocType.str_repeat('[]', $arrayDeepCount);
                $targetType = 'array';
                $isObjectType = !(RpcDefinition::getInstance()->getStruct($childType) instanceof NotDefinitionStruct);
            }
        } elseif (RpcDefinition::getInstance()->inStructCache($targetType)) {
            $isObjectType = !(RpcDefinition::getInstance()->getStruct($targetType) instanceof NotDefinitionStruct);

            $docType = $targetType = $this->getFullStructNameByStruct(
                RpcDefinition::getInstance()->getStruct($targetType)
            );
        }

        return [$targetType, $docType, $isObjectType];
    }

    private function getFullStructNameByStruct (Struct $struct): string {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));

        $className = class_basename($struct->name);
        /**
         * warp base name
         * [App\a\b\c\d\e\abc] -> b\c\d\e\abc
         */
        $simpleNamespace = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $struct->namespace);

        return "\Mainto\Bridge\Structs\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : "").'\\'.$className;
    }

    public function enum () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('enum_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        if (file_exists(path_join($dumpPath, "/micro-bridge/src/Enums/{$microName}"))) {
            remove_all(path_join($dumpPath, "/micro-bridge/src/Enums/{$microName}"));
        }

        foreach ($this->document->enums as $enum) {
            /**
             * warp base name
             * [App\a\b\c\Enum\d\e\abcEnum] -> b\c\d\e\abcEnum
             * [App\a\Enum\d\eabcEnum] -> d\e\abcEnum
             */
            $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $enum->namespace);
            $currentName = preg_replace('/\\\\?Enum\\\\?/', '\\', $currentName);
            $simpleNamespace = trim($currentName, '\\');

            $className = class_basename($enum->name);
            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Enums/{$microName}",
                str_replace('\\', '/', $simpleNamespace),
                "$className.php"
            );
            if (file_exists($path)) {
                $this->warn(
                    sprintf("enum path is duplicate after resolve namespace, origin [%s] -> [%s]",
                        $enum->namespace.'\\'.$className,
                        $simpleNamespace.'\\'.$className
                    ));
            }

            file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
            file_put_contents($path, "<?php\n".view('enum', [
                    'service'   => 'Bridge',
                    'microName' => $microName,
                    'className' => $className,
                    'constants' => $enum->enums,
                    'namespace' => "Mainto\Bridge\Enums\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : ""),
                ])->render());
        }
    }

    public function packageJson () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));
        file_put_contents($dumpPath."/micro-bridge/composer.json", json_encode([
            'name'        => 'mainto/micro-bridge-'.str_replace("_", "-", snake_case($microName)),
            'homepage'    => 'https://code.hzmantu.com',
            'description' => "PHP {$microName} Serivce RPC Bridge from MainTo Company",
            'keywords'    => ["micro core bridge {$microName} sdk mainto"],
            'license'     => ['MIT'],
            'require'     => ['php' => '^7.2.1'],
            'autoload'    => [
                'psr-4' => [
                    "Mainto\\Bridge\\Enums\\{$microName}\\"   => "src/Enums/{$microName}",
                    "Mainto\\Bridge\\Invokes\\{$microName}\\" => "src/Invokes/{$microName}",
                    "Mainto\\Bridge\\Structs\\{$microName}\\" => "src/Structs/{$microName}",
                ],
            ],
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }

    public function struct () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('struct_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        if (file_exists(path_join($dumpPath, "/micro-bridge/src/Structs/{$microName}"))) {
            remove_all(path_join($dumpPath, "/micro-bridge/src/Structs/{$microName}"));
        }

        foreach ($this->document->structs as $struct) {
            if ($struct instanceof NotDefinitionStruct) {
                continue;
            }

            if ($struct->namespace === RpcDefinition::EmptyType) {
                continue;
            }

            /**
             * warp base name
             * [App\a\b\c\d\e\abc] -> b\c\d\e\abc
             */
            $simpleNamespace = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $struct->namespace);

            $className = class_basename($struct->name);
            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Structs/{$microName}",
                str_replace('\\', '/', $simpleNamespace),
                "$className.php"
            );

            if (file_exists($path)) {
                $this->warn(
                    sprintf("enum path is duplicate after resolve namespace, origin [%s] -> [%s]",
                        $struct->namespace.'\\'.$className,
                        $simpleNamespace.'\\'.$className
                    ));
            }

            file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);

            file_put_contents($path, "<?php\n".view('object', [
                    'object' => [
                        'namespace'  => "Mainto\Bridge\Structs\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : ""),
                        'name'       => $className,
                        'properties' => array_map(function (Property $property) {
                            [$type, $docType] = $this->parserTypeString($property->type);

                            if (isset(Language::$rpcTypeMap[$type])) {
                                $type = Language::$rpcTypeMap[$type];
                            }

                            if (isset(Language::$rpcDocTypeMap[$docType]) && Language::$rpcDocTypeMap[$docType] != $docType) {
                                $docType = Language::$rpcDocTypeMap[$docType]." // origin: ".$docType;
                            }

                            return [
                                'type'                  => $type,
                                'doc_type'              => $docType,
                                'name'                  => $property->name,
                                'defaultValueAvailable' => $property->defaultValueAvailable,
                                'default'               => $property->default,
                                'nullable'              => $property->nullable,
                            ];
                        }, $struct->properties),
                    ],
                ])->render());
        }
    }

    private function saveStruct ($saveDir, $namespace, Struct $struct): string {
        if ($struct->namespace === RpcDefinition::EmptyType) {
            return "";
        }
        if ($struct instanceof NotDefinitionStruct) {
            return "";
        }

        if (isset($this->objectMap[$namespace.$struct->name])) {
            return $this->objectMap[$namespace.$struct->name];
        }

        $this->objectMap[$namespace.$struct->name] = $namespace.'\\'.$struct->name;

        $dumpObject = [
            'namespace'  => $namespace,
            'name'       => $struct->name,
            'properties' => array_map(function (Property $property) use ($namespace, $saveDir) {
                $type = $docType = $property->type;

                [$newType, $newDocType] = $this->saveType($saveDir, $namespace, $type, $docType);

                if (isset(Language::$rpcTypeMap[$newType])) {
                    $newType = Language::$rpcTypeMap[$newType];
                }

                if (isset(Language::$rpcDocTypeMap[$newDocType]) && Language::$rpcDocTypeMap[$newDocType] != $newDocType) {
                    $newDocType = Language::$rpcDocTypeMap[$newDocType]." // origin: ".$newDocType;
                }

                return [
                    'type'                  => $newType ?: $type,
                    'doc_type'              => $newDocType ?: $docType,
                    'name'                  => $property->name,
                    'defaultValueAvailable' => $property->defaultValueAvailable,
                    'default'               => $property->default,
                    'nullable'              => $property->nullable,
                ];
            }, $struct->properties),
        ];

        $path = path_join(
            $saveDir,
            "$struct->name.php"
        );

        file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
        file_put_contents($path, "<?php\n".view('object', ['object' => $dumpObject])->render());

        return $this->objectMap[$namespace.$struct->name];
    }

    private function saveType ($saveDir, $namespace, $targetType, $docType): array {
        if (starts_with($targetType, "Map<")) {
            preg_match('/Map<(.*)>/', $targetType, $match);
            $subtype = $this->saveStruct(
                $saveDir, $namespace,
                RpcDefinition::getInstance()->getStruct(ltrim(explode(',', $match[1])[1]))
            );

            $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", '\\'.ltrim($subtype, '\\'));
            $targetType = "\Mainto\RpcServer\Util\Types\Map";
        } else if (ends_with($targetType, "[]")) {
            [$childType, $arrayDeepCount] = ArrayTypeParser::getArrayType($targetType);
            $docType = $childType.str_repeat('[]', $arrayDeepCount);
            if (RpcDefinition::getInstance()->inStructCache($childType)) {
                $childDocType = $this->saveStruct($saveDir, $namespace, RpcDefinition::getInstance()->getStruct($childType));
                $docType = $childDocType.str_repeat('[]', $arrayDeepCount);
                $targetType = 'array';
            }
        } elseif (RpcDefinition::getInstance()->inStructCache($docType)) {
            $targetType = $docType = $this->saveStruct($saveDir, $namespace, RpcDefinition::getInstance()->getStruct($docType));
        }

        return [$targetType, $docType];
    }
}
