<?php

namespace Mainto\RpcTool\Command;

use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\Definition\Method;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcRouter;
use ReflectionClass;
use ReflectionException;
use RuntimeException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Throwable;

class RpcMakeControllerTestClass extends Command {

    /**
     * ===========================================================
     * Dump IDE End
     */

    public string $testNamespaceBase;
    public string $controllerNamespace;
    public string $testPathBase;

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws ReflectionException
     * @throws Throwable
     */
    public function handle () {
        $className = $this->argument('class_name');
        $this->testNamespaceBase = trim($this->option('test_class_namespace_base'));
        $this->controllerNamespace = strval($this->option('controller_namespace_base'));
        $this->testPathBase = format_path($this->option('output_dir'));
        if ($className === '*') {
            $classes = RpcDefinition::getInstance()->controllers;
        } else {
            if (!isset(RpcDefinition::getInstance()->controllers[RpcRouter::getInstance()->getControllerRegisterName($this->controllerNamespace.'\\'.$className)])) {
                $className = rtrim($className, 'Controller');
                $choices = [];
                foreach (RpcDefinition::getInstance()->controllers as $register => $controller) {
                    if (Str::contains($register, $className) || Str::contains($className, $register)) {
                        $choices[] = $register;
                    }
                }
                $choice = $this->choice('请选择对应的控制器', $choices, 0);
                if (!$choice) {
                    throw new RuntimeException("The class $className is not defined in rpc route! ");
                } else {
                    $classes = [RpcDefinition::getInstance()->controllers[$choice]];
                }
            } else {
                $classes = [RpcDefinition::getInstance()->controllers[RpcRouter::getInstance()->getControllerRegisterName($this->controllerNamespace.'\\'.$className)]];
            }
        }

        foreach ($classes as $class) {
            $namespace = explode('\\', str_replace($this->controllerNamespace, '', get_class($class->getInstance())));
            $unset_controller_name = array_slice($namespace, 0, count($namespace) - 1);
            $class_name = array_pop($namespace);
            $test_namespace = implode('\\', $unset_controller_name);
            $dir_name = path_join($this->testPathBase, implode('/', $unset_controller_name));
            $file_name = $dir_name.'/'.$class_name.'Test.php';
            !file_exists($dir_name) and mkdir($dir_name, 0777, true);
            $rpcObjects = [];
            foreach ($class->methods as $method) {
                if (parameters_object($method->getParameters()) && !in_array($method->requestType, $rpcObjects)) {
                    $rpcObjects[] = $method->requestType;
                }
            }

            if (!file_exists($file_name)) {
                file_put_contents($file_name, "<?php\n".view('test-controller', [
                        'controllerDocNamespace'  => config('rpc-tool.cmd.namespace.controller_doc'),
                        'class_name'              => $class_name,
                        'controllerNamespaceBase' => $this->controllerNamespace.($test_namespace ? $test_namespace."\\".$class_name : '\\'.$class_name),
                        'methods'                 => $class->methods,
                        'rpcObjects'              => $rpcObjects,
                        'namespace'               => $this->testNamespaceBase.($test_namespace ?: ''),
                        'package'                 => '@package '.$this->testNamespaceBase.($test_namespace ?: ''),
                    ])->render());
            } else {
                $this->rebuildTestClassDoc($file_name, $class_name, $class->methods);
            }
        }

        return 0;
    }

    /**
     * @param $file
     * @param $class_name
     * @param array<Method> $methods
     * @throws ReflectionException
     */
    public function rebuildTestClassDoc ($file, $class_name, $methods) {
        $refClass = new ReflectionClass(get_class_from_file($file));
        $package = substr($refClass->name, 0, -(strlen($refClass->getShortName())) - 1);

        $doc = <<<EOF
/**
 * Class {$class_name}Test
 * @package {$package}
 *
 * --------------------------------------
 * auto gen method annotations, if the method name is already exists,
 * please use getResponse([method_name], [params])
 * --------------------------------------
EOF;
        foreach ($methods as $method) {
            if (!parameters_object($method->getParameters())) {
                $doc .= "\n * @method call".Str::ucfirst($method->name)."(array \$params = [], bool \$enable_doc = true)";
            } else {
                $doc .= "\n * @method call".Str::ucfirst($method->name)."(array|".class_basename($method->requestType)." \$params = [], bool \$enable_doc = true)";
            }
        }
        $doc .= "\n */";

        file_put_contents($file, str_replace($refClass->getDocComment(), $doc, file_get_contents($file)));
    }

    protected function configure () {
        parent::configure();
        $this->setName('rpc:make-controller-test')
            ->setDescription('Create a new controller test class ')
            ->addArgument('class_name', InputArgument::REQUIRED, 'The registered controller name')
            ->addOption('test_class_namespace_base', 't', InputOption::VALUE_OPTIONAL, 'The test class namespace base', 'Tests\\')
            ->addOption('controller_namespace_base', 'c', InputOption::VALUE_OPTIONAL, 'The controller namespace base', 'App\\')
            ->addOption('output_dir', 'o', InputOption::VALUE_OPTIONAL, 'The test class output dir', './tests/');
    }
}
