<?php


namespace Mainto\RpcTool\Command;


use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use Mainto\RpcTool\Document\Document;

class RpcSdkPHPCommand extends Command {

    protected $signature = 'rpc:sdk-php {option?} {--write_json=} {--dump_path=} {--enum_namespace=} {--enum_dump_path=} {--controller_namespace=} {--controller_dump_path=} {--struct_namespace=} {--struct_dump_path=}';
    private array $objectMap = [];

    private Document $document;

    public function handle () {
        $this->document = Document::fromRpcDefinition(RpcDefinition::getInstance());
        $this->document->resolveEnumAndException();

        switch ($this->argument('option')) {
            case 'controller':
                $this->controller();
                break;
            case 'enum':
                $this->enum();
                break;
            default:
                $this->controller();
                $this->enum();
        }
        if ($this->option('write_json')) {
            $this->packageJson();
        }
    }

    public function packageJson () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));
        file_put_contents($dumpPath."/micro-bridge/composer.json", json_encode([
            'name'        => 'mainto/micro-bridge-'.str_replace("_", "-", snake_case($microName)),
            'homepage'    => 'https://code.hzmantu.com',
            'description' => "PHP {$microName} Serivce RPC Bridge from MainTo Company",
            'keywords'    => ["micro core bridge {$microName} sdk mainto"],
            'license'     => ['MIT'],
            'require'     => ['php' => '^7.2.1'],
            'autoload'    => [
                'psr-4' => [
                    "Mainto\\Bridge\\Enums\\{$microName}\\"   => "src/Enums/{$microName}",
                    "Mainto\\Bridge\\Invokes\\{$microName}\\" => "src/Invokes/{$microName}",
                    "Mainto\\Bridge\\Objects\\{$microName}\\" => "src/Objects/{$microName}",
                ],
            ],
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }

    public function controller () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        foreach ($this->document->controllers as $controller) {
            $registerPath = str_replace('\\', '/', $controller->registerName);
            $controllerDir = $microName;
            if (strrpos($registerPath, "/")) {
                $controllerDir = path_join($controllerDir, dirname($registerPath));
            }

            $namespace = str_replace("/", "\\", $controllerDir);

            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Invokes/",
                $controllerDir,
                basename($registerPath).".php"
            );
            if (!file_exists(dirname($path))) {
                mkdir(dirname($path), 0777, true);
            }
            foreach ($controller->methods as $method) {
                if (RpcDefinition::getInstance()->inStructCache($method->requestType)) {
                    $this->saveStruct(
                        path_join(dirname($path), $controller->name, safe_namespace(Str::studly($method->name))),
                        'Mainto\Bridge\Invokes\\'.$namespace.'\\'.basename($registerPath).'\\'.safe_namespace(Str::studly($method->name)),
                        RpcDefinition::getInstance()->getStruct($method->requestType)
                    );
                    if ($method->responseType !== "mixed" && !in_array($method->responseType, Language::$simpleReturnTypeMap)) {
                        $this->saveType(
                            path_join(dirname($path), $controller->name, safe_namespace(Str::studly($method->name))),
                            'Mainto\Bridge\Invokes\\'.$namespace.'\\'.basename($registerPath).'\\'.safe_namespace(Str::studly($method->name)),
                            $method->responseType,
                            $method->responseType,
                        );
                    }
                }
            }

            file_put_contents($path, "<?php\n".view('bridge-controller', [
                    'microName'          => $microName,
                    'className'          => $controller->registerName,
                    'shortClassName'     => class_basename($controller->registerName),
                    'methods'            => $controller->methods,
                    'namespace'          => $namespace,
                    'mockClassNamespace' => config('rpc-tool.cmd.namespace.mock_class'),
                    'rpcClassNamespace'  => config('rpc-tool.cmd.namespace.rpc_class'),
                ])->render());
        }
    }

    public function enum () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $enumNamespace = $this->option('enum_namespace') ?: 'App\Enums';
        $dumpPath = value_if($this->option('enum_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        /**
         * warp base name
         * [App\a\b\c\Enum\d\e\abcEnum] -> b\c\d\e\abcEnum
         * [App\a\Enum\d\eabcEnum] -> d\e\abcEnum
         */

        $pregWarp = function ($preg, $replace, $currentName) {
            return preg_replace($preg, $replace, $currentName);
        };

        if (file_exists(path_join($dumpPath, "/micro-bridge/src/Enums/{$microName}"))) {
            remove_all(path_join($dumpPath, "/micro-bridge/src/Enums/{$microName}"));
        }

        foreach ($this->document->enums as $enum) {
            $currentName = $pregWarp('/.*[aA]pp\\\\[\w]+\\\\?/', '', $enum->namespace);
            $currentName = $pregWarp('/\\\\?Enum\\\\?/', '\\', $currentName);
            $simpleNamespace = trim($currentName, '\\');

            $className = class_basename($enum->name);
            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Enums/{$microName}",
                str_replace('\\', '/', $simpleNamespace),
                "$className.php"
            );
            if (file_exists($path)) {
                $this->warn(
                    sprintf("enum path is duplicate after resolve namesapce, origin [%s] -> [%s]",
                        $enum->namespace.'\\'.$className,
                        $simpleNamespace.'\\'.$className
                    ));
            }

            file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
            file_put_contents($path, "<?php\n".view('enum', [
                    'service'   => 'Bridge',
                    'microName' => $microName,
                    'className' => $className,
                    'constants' => $enum->enums,
                    'namespace' => "Mainto\Bridge\Enums\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : ""),
                ])->render());
        }
    }

    private function saveType ($saveDir, $namespace, $targetType, $docType): array {
        if (starts_with($targetType, "Map<")) {
            preg_match('/Map<(.*)>/', $targetType, $match);
            $subtype = $this->saveStruct(
                $saveDir, $namespace,
                RpcDefinition::getInstance()->getStruct(ltrim(explode(',', $match[1])[1]))
            );

            $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", '\\'.ltrim($subtype, '\\'));
            $targetType = "\Mainto\RpcServer\Util\Types\Map";
        } else if (ends_with($targetType, "[]")) {
            [$childType, $arrayDeepCount] = ArrayTypeParser::getArrayType($targetType);
            $docType = $childType.str_repeat('[]', $arrayDeepCount);
            if (RpcDefinition::getInstance()->inStructCache($childType)) {
                $childDocType = $this->saveStruct($saveDir, $namespace, RpcDefinition::getInstance()->getStruct($childType));
                $docType = $childDocType.str_repeat('[]', $arrayDeepCount);
                $targetType = 'array';
            }
        } elseif (RpcDefinition::getInstance()->inStructCache($docType)) {
            $targetType = $docType = $this->saveStruct($saveDir, $namespace, RpcDefinition::getInstance()->getStruct($docType));
        }

        return [$targetType, $docType];
    }

    private function saveStruct ($saveDir, $namespace, Struct $struct): string {
        if ($struct->namespace === RpcDefinition::EmptyType) {
            return "";
        }
        if ($struct instanceof NotDefinitionStruct) {
            return "";
        }

        if (isset($this->objectMap[$namespace.$struct->name])) {
            return $this->objectMap[$namespace.$struct->name];
        }

        $this->objectMap[$namespace.$struct->name] = $namespace.'\\'.$struct->name;

        $dumpObject = [
            'namespace'  => $namespace,
            'name'       => $struct->name,
            'properties' => array_map(function (Property $property) use ($namespace, $saveDir) {
                $type = $docType = $property->type;

                [$newType, $newDocType] = $this->saveType($saveDir, $namespace, $type, $docType);

                if (isset(Language::$rpcTypeMap[$newType])) {
                    $newType = Language::$rpcTypeMap[$newType];
                }

                if (isset(Language::$rpcDocTypeMap[$newDocType]) && Language::$rpcDocTypeMap[$newDocType] != $newDocType) {
                    $newDocType = Language::$rpcDocTypeMap[$newDocType]." // origin: ".$newDocType;
                }

                return [
                    'type'                  => $newType ?: $type,
                    'doc_type'              => $newDocType ?: $docType,
                    'name'                  => $property->name,
                    'defaultValueAvailable' => $property->defaultValueAvailable,
                    'default'               => $property->default,
                    'nullable'              => $property->nullable,
                ];
            }, $struct->properties),
        ];

        $path = path_join(
            $saveDir,
            "$struct->name.php"
        );

        file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
        file_put_contents($path, "<?php\n".view('object', ['object' => $dumpObject])->render());

        return $this->objectMap[$namespace.$struct->name];
    }
}
