<?php

use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\Definition\Parameter;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcObject;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;

if (!function_exists('value_if')) {
    function value_if ($first, $default) {
        return $first ?: $default;
    }
}

if (!function_exists('short_type')) {
    function short_type ($type): string {
        if (in_array($type, Language::$simpleType)) {
            return $type;
        }

        return class_basename($type);
    }
}

if (!function_exists('is_language_type')) {
    function is_language_type ($type): bool {
        return in_array($type, Language::$simpleType);
    }
}

if (!function_exists('response_is_object')) {
    function response_is_object ($type): string {
        if (Str::endsWith($type, '[]')) {
            [$subChild, $count] = ArrayTypeParser::getArrayType($type);
            if (RpcDefinition::getInstance()->inStructCache($subChild)) {
                return true;
            }
        }

        if (RpcDefinition::getInstance()->inStructCache($type)) {
            return true;
        }

        return false;
    }
}

if (!function_exists('response_type')) {
    function response_type ($methodName, $type): string {
        if (Str::endsWith($type, '[]')) {
            [$subChild, $count] = ArrayTypeParser::getArrayType($type);
            if (RpcDefinition::getInstance()->inStructCache($subChild)) {
                return Str::studly($methodName).'\\'.RpcDefinition::getInstance()->getStruct($subChild)->name. str_repeat('[]', $count);
            }
        }

        if (RpcDefinition::getInstance()->inStructCache($type)) {
            return Str::studly($methodName).'\\'.RpcDefinition::getInstance()->getStruct($type)->name;
        }

        return $type;
    }
}

if (!function_exists('parameters_to_str')) {
    function parameters_to_str ($methodName, $parameters): string {
        $argStr = "";
        /** @var Parameter $parameter */
        foreach ($parameters as $name => $parameter) {
            if ($parameter->type == RpcContext::class) continue;
            if (is_subclass_of($parameter->type, RpcObject::class)) {
                return ($parameter->nullable ? '?' : '')
                    .Str::studly($methodName).'\\'.RpcDefinition::getInstance()->getStruct($parameter->type)->name.' $'.$parameter->name
                    . ($parameter->defaultValueAvailable ? ' = '.var_export_min($parameter->default) : '');
            }

            $argStr .= (Language::$rpcTypeMap[$parameter->type] ?? $parameter->type).' $'.$parameter->name;
            $argStr .= $parameter->defaultValueAvailable ? ' = '.var_export_min($parameter->default) : '';
            $argStr .= ', ';
        }

        return rtrim($argStr, ', ');
    }
}

if (!function_exists('full_type_name')) {
    function full_type_name ($type) {
        if (Str::endsWith($type, '[]')) {
            $subType = substr($type, 0, -2);
            if (class_exists($subType)) {
                return '\\'.ltrim($subType, '\\').'[]';
            } else {
                return $type;
            }
        }
        if (class_exists($type)) {
            return '\\'.ltrim($type, '\\');
        }

        return $type;
    }
}

if (!function_exists('parameters_to_array')) {
    function parameters_to_array ($parameters): array {
        $array = [];
        /** @var Parameter $parameter */
        foreach ($parameters as $name => $parameter) {
            if ($parameter->type == RpcContext::class) continue;
            if (is_subclass_of($parameter->type, RpcObject::class)) {
                return [$parameter->name => $parameter];
            }
            $array[$parameter->name] = $parameter;
        }

        return $array;
    }
}

if (!function_exists('parameters_object')) {
    function parameters_object ($parameters): ?Parameter {
        /** @var Parameter $parameter */
        foreach ($parameters as $name => $parameter) {
            if ($parameter->type == RpcContext::class) continue;
            if (is_subclass_of($parameter->type, RpcObject::class)) {
                return $parameter;
            }
        }

        return null;
    }
}

if (!function_exists('str_camel')) {
    function str_camel ($str): string {
        return Str::camel($str);
    }
}

if (!function_exists('var_export_min')) {
    /**
     * dump var
     *
     * @param $var
     * @param bool $return
     * @return mixed|string
     */
    function var_export_min ($var, $return = true): string {
        if (is_array($var)) {
            $toImplode = [];
            foreach ($var as $key => $value) {
                $toImplode[] = var_export($key, true).' => '.var_export_min($value, true);
            }
            $code = '['.implode(',', $toImplode).']';
            if ($return) return $code;
            else echo $code;
        } else {
            // null var_export 会变成 NULL
            if ($var === null) {
                if ($return) return 'null';
                else echo $var;
            }
            return var_export($var, $return);
        }
    }
}

if (!function_exists('del_file_tree')) {
    /**
     * Delete file tree
     *
     * @param $dir
     * @return bool
     */
    function del_file_tree ($dir): bool {
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            (is_dir("$dir/$file")) ? del_file_tree("$dir/$file") : unlink("$dir/$file");
        }
        return rmdir($dir);
    }
}

if (!function_exists('get_files')) {
    /**
     * Get files
     * @param $dir
     * @return array
     */
    function get_files ($dir): array {
        $files = [];
        $scan = scandir($dir);
        foreach ($scan as $item) {
            if ($item == '.' || $item == '..') continue;
            if (is_dir($dir.'/'.$item)) {
                $files = array_merge($files, get_files($dir.'/'.$item));
            } else {
                $files[] = $dir.'/'.$item;
            }
        }
        return $files;
    }
}

if (!function_exists('format_path')) {
    /**
     * Format the relative path as an absolute path with the current working directory as the reference
     *
     * @param $path
     * @return string
     */
    function format_path ($path): string {
        if ($path[0] == ".") {
            $path = getcwd()."/{$path}";
        }

        $path = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path);
        $parts = array_filter(explode(DIRECTORY_SEPARATOR, $path), 'strlen');
        $absolutes = [];
        foreach ($parts as $part) {
            if ('.' == $part) continue;
            if ('..' == $part) {
                array_pop($absolutes);
            } else {
                $absolutes[] = $part;
            }
        }
        return "/".implode(DIRECTORY_SEPARATOR, $absolutes);
    }
}

if (!function_exists('safe_url_string')) {
    /**
     * replace the str except [\w,-,.] to '-'
     *
     * @param string $str
     * @return string
     */
    function safe_url_string (string $str): string {
        return preg_replace('/[^\w\-_.]/m', '-', $str);
    }
}
