<?php
/**
 * Created by PhpStorm.
 * User: liuzaisen
 * Date: 2018/10/18
 * Time: 8:09 PM
 */

namespace Mainto\RpcTool\Command;


use GuzzleHttp\Client;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcTool\Document\Document;

class RpcDocUpload extends Command {
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:doc_upload {--file_path=} {--doc_dir=} {--set_version=} {--base_path=} {--composer_path=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'upload document';

    /**
     * @var Document
     */
    protected Document $document;

    /**
     * RpcDocUpload constructor.
     */
    public function __construct () {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @throws \Exception
     */
    public function handle () {
        $this->initDocument();
        $this->setVersion();
        $this->upload();
    }

    /**
     * 初始化文档信息
     */
    private function initDocument () {
        if (!$file_path = $this->option('file_path')) {
            $file_path = config('rpc-tool.micro_doc.doc_dir').'/definition.json';
        }

        if (file_exists($file_path)) {
            $this->document = unserialize(file_get_contents($file_path));
        } else {
            $this->document = Document::fromRpcDefinition(RpcDefinition::getInstance());
        }
    }

    /**
     * 设置版本信息
     */
    private function setVersion () {
        $version = $this->option('set_version');

        if (!$version) {
            if (!$set_version = env('CI_COMMIT_REF_SLUG')) {
                if ($base_path = $this->option('base_path')) {
                    $git_file = $base_path.'/.git/HEAD';
                } else {
                    $git_file = base_path('/.git/HEAD');
                }

                if (!file_exists($git_file)) {
                    throw new \RuntimeException('get version fail');
                }

                $set_version = trim(str_replace(['ref: refs/heads/'], '', file_get_contents($git_file)));
                if (!$set_version) throw new \RuntimeException('get version fail');
            }
            $version = $set_version;
        }

        $this->document->version = safe_url_string($version);
    }

    /**
     * 上传文档
     */
    private function upload () {
        $skip = env('UPLOAD_SKIP_CONFIRM');
        $versionText = "{$this->document->serviceName}/{$this->document->version}";
        $domain = config('rpc-tool.micro_doc.doc_domain');

        if ($skip) {
            for ($i = 5; $i > 0; $i--) {
                $this->comment("\rstart upload to {$versionText} in $i seconds...");
                sleep(1);
            }
        } elseif (!$this->confirm("upload to {$versionText}?")) {
            $this->comment("cancel!");
            return;
        }

        [$docPaths, $docFiles] = $this->getDocs();
        $this->document->docPaths = $docPaths;
        $this->document->dependencies = $this->getDependencies();

        $this->comment("uploading...");
        $client = new Client();

        $multi = [
            [
                'name'     => 'data',
                'contents' => json_encode($this->document->toArray(), JSON_UNESCAPED_UNICODE),
            ],
        ];

        foreach ($docFiles as $docFile) {
            $multi[] = [
                'name'     => str_replace(realpath(app_path('../doc')), '', $docFile),
                'contents' => fopen($docFile, 'r'),
                'filename' => $docFile,
            ];
        }

        $res = $client->post("{$domain}/@/api/service/version/upload", [
            'multipart' => $multi,
        ]);

        $this->comment((string)$res->getBody());
    }

    private function getDependencies () {
        $composeFile = app_path('../composer.json');
        if ($this->option('composer_path')) {
            $composeFile = $this->option('composer_path');
        }

        if (file_exists($composeFile)) {
            $composeJson = json_decode(file_get_contents($composeFile), true);
            // substr mainto/micro-bridge-
            return array_map(fn($item) => Str::studly(substr($item, 20)),
                array_filter(array_keys(
                    $composeJson['require']),
                    fn($item) => Str::startsWith($item, "mainto/micro-bridge-")
                )
            );
        } else {
            throw new \RuntimeException("not found composer.json file");
        }
    }

    /**
     * 处理文档
     */
    private function getDocs (): array {
        $doc_dir = $this->option('doc_dir');
        if (!$doc_dir) $doc_dir = base_path('doc');

        $doc = [];
        $files = [];

        if (file_exists($doc_dir)) {
            $docFiles = get_files($doc_dir);
            foreach ($docFiles as $filePath) {
                if (!Str::endsWith($filePath, '.md')) {
                    $files[] = $filePath;
                    continue;
                };

                $path = str_replace($doc_dir, "", $filePath);
                $content = file_get_contents($filePath);
                $matches = [];
                if (preg_match_all('/.*!\[.*]\((.*)\).*/', $content, $matches)) {
                    foreach ($matches[0] as $index => $origin) {
                        if (Str::startsWith($matches[1][$index], 'http') || Str::startsWith($matches[1][$index], 'https')) {
                            continue;
                        }
                        $imagePath = path_join(
                            "asserts/doc_image",
                            $this->document->serviceName,
                            $this->document->version,
                            dirname(str_replace(realpath(app_path('../doc')), '', $filePath)),
                            $matches[1][$index],
                        );

                        $content = str_replace($origin, str_replace($matches[1][$index], $imagePath, $origin), $content);
                    }
                }

                $doc[] = [
                    'path'    => $path,
                    'content' => $content,
                ];
            }
        }

        return [$doc, $files];
    }
}
