<?php


namespace Mainto\RpcTool\Command;


use Illuminate\Console\Command;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\Util\Language;

class RpcSdkPHPCommand extends Command {

    protected $signature = 'rpc:sdk-php {option?} {--write_json=} {--enum_namespace=} {--enum_dump_path=} {--controller_namespace=} {--controller_dump_path=} {--struct_namespace=} {--struct_dump_path=}';
    private array $objectMap = [];

    public function handle () {
        switch ($this->argument('option')) {
            case 'controller':
                $this->controller();
                break;
            case 'enum':
                $this->enum();
                break;
            case 'struct':
                $this->struct();
                break;
            default:
                $this->controller();
                $this->enum();
                $this->struct();
        }
        if ($this->option('write_json')) {
            $this->packageJson();
        }
    }

    public function packageJson() {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = $this->option('controller_dump_path') ?: format_path('.');
        file_put_contents($dumpPath."/micro-bridge/composer.json", json_encode([
            'name'        => 'mainto/micro-bridge-'.str_replace("_", "-", snake_case($microName)),
            'homepage'    => 'https://code.hzmantu.com',
            'description' => "PHP {$microName} Serivce RPC Bridge from MainTo Company",
            'keywords'    => ["micro core bridge {$microName} sdk mainto"],
            'license'     => ['MIT'],
            'require'     => ['php' => '^7.2.1'],
            'autoload'    => [
                'psr-4' => [
                    "Mainto\\Bridge\\Enums\\{$microName}\\"   => "src/Enums/{$microName}",
                    "Mainto\\Bridge\\Invokes\\{$microName}\\" => "src/Invokes/{$microName}",
                    "Mainto\\Bridge\\Objects\\{$microName}\\" => "src/Objects/{$microName}",
                ],
            ],
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }

    public function controller () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = $this->option('controller_dump_path') ?: format_path('.');
        $controllerNamespace = $this->option('controller_namespace') ?: "App\Rpc\Controllers";

        foreach (RpcDefinition::getInstance()->controllers as $controller) {
            $namespacePath = str_replace('\\', '/', str_replace($controllerNamespace, "", $controller->namespace));

            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Invokes/",
                $microName,
                $namespacePath,
                class_basename($controller->registerName).".php"
            );
            if (!file_exists(dirname($path))) {
                mkdir(dirname($path), 0777, true);
            }
            file_put_contents($path, "<?php\n".view('bridge-controller', [
                    'microName'          => $microName,
                    'className'          => $controller->registerName,
                    'shortClassName'     => class_basename($controller->registerName),
                    'methods'            => $controller->methods,
                    'namespace'          => str_replace("/", "\\", path_join($microName, $namespacePath)),
                    'mockClassNamespace' => config('rpc-tool.cmd.namespace.mock_class'),
                    'rpcClassNamespace'  => config('rpc-tool.cmd.namespace.rpc_class'),
                ])->render());
        }
    }

    public function enum () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $enumNamespace = $this->option('enum_namespace') ?: 'App\Enums';
        $dumpPath = $this->option('enum_dump_path') ?: format_path('.');

        foreach (RpcDefinition::getInstance()->enums as $enum) {
            $simpleNamespace = trim(str_replace($enumNamespace, '', $enum->namespace), '/\\');
            $className = class_basename($enum->name);
            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Enums/{$microName}",
                str_replace('\\', '/', $simpleNamespace),
                "$className.php"
            );
            file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
            file_put_contents($path, "<?php\n".view('enum', [
                    'service'   => 'Bridge',
                    'microName' => $microName,
                    'className' => $className,
                    'constants' => $enum->enums,
                    'namespace' => "Mainto\Bridge\Enums\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : ""),
                ])->render());
        }
    }

    public function struct () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $controllerNamespace = $this->option('struct_namespace') ?: "App\Rpc";
        $dumpPath = $this->option('struct_dump_path') ?: format_path('.');

        foreach (RpcDefinition::getInstance()->structs as $struct) {
            $this->genStruct($microName, $controllerNamespace, $dumpPath, $struct);
        }
    }

    private function genStruct ($microName, $controllerNamespace, $dumpPath, Struct $struct): string {
        if ($struct->namespace === RpcDefinition::EmptyType) {
            return "";
        }
        if ($struct instanceof NotDefinitionStruct) {
            return "";
        }
        if (isset($this->objectMap[$struct->namespace.'\\'.$struct->name])) {
            return $this->objectMap[$struct->namespace.'\\'.$struct->name];
        }

        $this->objectMap[$struct->namespace.'\\'.$struct->name] = $newName = "Mainto\Bridge\Objects\\$microName".str_replace($controllerNamespace, '', $struct->namespace).'\\'.$struct->name;

        $dumpObject = [
            'namespace'  => "Mainto\Bridge\Objects\\$microName".str_replace($controllerNamespace, '', $struct->namespace),
            'name'       => $struct->name,
            'properties' => array_map(function (Property $property) use ($dumpPath, $controllerNamespace, $microName) {
                $docType = $property->type;
                if (isset(Language::$rpcTypeMap[$property->type])) {
                    $docType = Language::$rpcTypeMap[$property->type];
                }

                if (starts_with($property->type, "Map::")) {
                    $subtype = $this->genStruct(
                        $microName,
                        $controllerNamespace,
                        $dumpPath,
                        RpcDefinition::getInstance()->getStruct(substr($property->type, 5))
                    );

                    $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", '\\'.ltrim($subtype, '\\'));
                }

                if (ends_with($property->type, "[]")) {
                    $docType = $property->type;
                }

                if (class_exists($property->type)) {
                    $docType = '\\'.ltrim($property->type, '\\');;
                }

                return [
                    'doc_type'              => $docType,
                    'type'                  => $this->propertyType($property->type),
                    'name'                  => $property->name,
                    'defaultValueAvailable' => $property->defaultValueAvailable,
                    'default'               => $property->default,
                    'nullable'              => $property->nullable,
                ];
            }, $struct->properties),
        ];

        $path = path_join(
            $dumpPath,
            "/micro-bridge/src/Objects/{$microName}",
            str_replace('\\', '/', str_replace($controllerNamespace, '', $struct->namespace)),
            "$struct->name.php"
        );

        file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
        file_put_contents($path, "<?php\n".view('object', ['object' => $dumpObject])->render());

        return $newName;
    }

    public function propertyType ($type): string {
        if (isset(Language::$rpcTypeMap[$type])) {
            return Language::$rpcTypeMap[$type];
        }

        if (starts_with($type, "Map::")) {
            return "\Mainto\RpcServer\Util\Types\Map";
        }

        if (ends_with($type, "[]")) {
            return 'array';
        }

        if (class_exists($type)) {
            return '\\'.ltrim($type, '\\');
        }

        return $type;
    }
}