<?php


namespace Mainto\RpcTool\Command;


use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\Util\Language;

class RpcSdkPHPCommand extends Command {

    protected $signature = 'rpc:sdk-php {option?} {--write_json=} {--dump_path=} {--enum_namespace=} {--enum_dump_path=} {--controller_namespace=} {--controller_dump_path=} {--struct_namespace=} {--struct_dump_path=}';
    private array $objectMap = [];

    public function handle () {
        switch ($this->argument('option')) {
            case 'controller':
                $this->controller();
                break;
            case 'enum':
                $this->enum();
                break;
            default:
                $this->controller();
                $this->enum();
        }
        if ($this->option('write_json')) {
            $this->packageJson();
        }
    }

    public function packageJson () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));
        file_put_contents($dumpPath."/micro-bridge/composer.json", json_encode([
            'name'        => 'mainto/micro-bridge-'.str_replace("_", "-", snake_case($microName)),
            'homepage'    => 'https://code.hzmantu.com',
            'description' => "PHP {$microName} Serivce RPC Bridge from MainTo Company",
            'keywords'    => ["micro core bridge {$microName} sdk mainto"],
            'license'     => ['MIT'],
            'require'     => ['php' => '^7.2.1'],
            'autoload'    => [
                'psr-4' => [
                    "Mainto\\Bridge\\Enums\\{$microName}\\"   => "src/Enums/{$microName}",
                    "Mainto\\Bridge\\Invokes\\{$microName}\\" => "src/Invokes/{$microName}",
                    "Mainto\\Bridge\\Objects\\{$microName}\\" => "src/Objects/{$microName}",
                ],
            ],
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }

    public function controller () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $dumpPath = value_if($this->option('controller_dump_path'), value_if($this->option('dump_path'), format_path('.')));
        $controllerNamespace = $this->option('controller_namespace') ?: "App\Rpc\Controllers";

        foreach (RpcDefinition::getInstance()->controllers as $controller) {
            $namespacePath = str_replace('\\', '/', str_replace($controllerNamespace, "", $controller->namespace));

            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Invokes/",
                $microName,
                $namespacePath,
                class_basename($controller->registerName).".php"
            );
            if (!file_exists(dirname($path))) {
                mkdir(dirname($path), 0777, true);
            }
            foreach ($controller->methods as $method) {
                if (RpcDefinition::getInstance()->inStructCache($method->requestType)) {
                    $this->saveStruct(
                        path_join(dirname($path), $controller->name, Str::studly($method->name)),
                        'Mainto\Bridge\Invokes\\'.str_replace("/", "\\", path_join($microName, $namespacePath)).'\\'.Str::studly($method->name),
                        RpcDefinition::getInstance()->getStruct($method->requestType)
                    );
                }
            }

            file_put_contents($path, "<?php\n".view('bridge-controller', [
                    'microName'          => $microName,
                    'className'          => $controller->registerName,
                    'shortClassName'     => class_basename($controller->registerName),
                    'methods'            => $controller->methods,
                    'namespace'          => str_replace("/", "\\", path_join($microName, $namespacePath)),
                    'mockClassNamespace' => config('rpc-tool.cmd.namespace.mock_class'),
                    'rpcClassNamespace'  => config('rpc-tool.cmd.namespace.rpc_class'),
                ])->render());
        }
    }

    public function enum () {
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));
        $enumNamespace = $this->option('enum_namespace') ?: 'App\Enums';
        $dumpPath = value_if($this->option('enum_dump_path'), value_if($this->option('dump_path'), format_path('.')));

        foreach (RpcDefinition::getInstance()->enums as $enum) {
            $simpleNamespace = trim(str_replace($enumNamespace, '', $enum->namespace), '/\\');
            $className = class_basename($enum->name);
            $path = path_join(
                $dumpPath,
                "/micro-bridge/src/Enums/{$microName}",
                str_replace('\\', '/', $simpleNamespace),
                "$className.php"
            );
            file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
            file_put_contents($path, "<?php\n".view('enum', [
                    'service'   => 'Bridge',
                    'microName' => $microName,
                    'className' => $className,
                    'constants' => $enum->enums,
                    'namespace' => "Mainto\Bridge\Enums\\$microName".($simpleNamespace ? '\\'.$simpleNamespace : ""),
                ])->render());
        }
    }

    private function saveStruct ($saveDir, $namespace, Struct $struct): string {
        if ($struct->namespace === RpcDefinition::EmptyType) {
            return "";
        }
        if ($struct instanceof NotDefinitionStruct) {
            return "";
        }

        if (isset($this->objectMap[$namespace.$struct->name])) {
            return $this->objectMap[$namespace.$struct->name];
        }

        $dumpObject = [
            'namespace'  => $namespace,
            'name'       => $struct->name,
            'properties' => array_map(function (Property $property) use ($namespace, $saveDir) {
                $type = $docType = $property->type;
                if (isset(Language::$rpcTypeMap[$property->type])) {
                    $docType = Language::$rpcTypeMap[$property->type];
                }

                if (starts_with($property->type, "Map<")) {
                    preg_match('/Map<(.*)>/', $property->type, $match);
                    $subtype = $this->saveStruct(
                        $saveDir, $namespace,
                        RpcDefinition::getInstance()->getStruct(ltrim(explode(',', $match[1])[1]))
                    );

                    $docType = sprintf("\Mainto\RpcServer\Util\Types\Map<%s>", '\\'.ltrim($subtype, '\\'));
                    $type = "\Mainto\RpcServer\Util\Types\Map";
                }

                if (ends_with($property->type, "[]")) {
                    $docType = $property->type;
                }

                if (RpcDefinition::getInstance()->inStructCache($docType)) {
                    $docType = $this->saveStruct($saveDir, $namespace, RpcDefinition::getInstance()->getStruct($docType));
                }

                return [
                    'type'                  => $type,
                    'doc_type'              => $docType,
                    'name'                  => $property->name,
                    'defaultValueAvailable' => $property->defaultValueAvailable,
                    'default'               => $property->default,
                    'nullable'              => $property->nullable,
                ];
            }, $struct->properties),
        ];

        $path = path_join(
            $saveDir,
            "$struct->name.php"
        );

        file_exists(pathinfo($path, PATHINFO_DIRNAME)) or mkdir(pathinfo($path, PATHINFO_DIRNAME), 0777, true);
        file_put_contents($path, "<?php\n".view('object', ['object' => $dumpObject])->render());

        return $this->objectMap[$namespace.$struct->name] = $namespace.'\\'.$struct->name;
    }
}
